% Generated by roxygen2 (4.0.2): do not edit by hand
\name{DiversityCurves}
\alias{DiversityCurves}
\alias{phyloDiv}
\alias{taxicDivCont}
\alias{taxicDivDisc}
\title{Diversity Curves}
\usage{
taxicDivCont(timeData, int.length = 1, int.times = NULL, plot = TRUE,
  plotLogRich = FALSE, timelims = NULL, drop.cryptic = FALSE)

taxicDivDisc(timeList, int.times = NULL, drop.singletons = FALSE,
  plot = TRUE, plotLogRich = FALSE, timelims = NULL,
  extant.adjust = 0.001, split.int = TRUE)

phyloDiv(tree, int.length = 0.1, int.times = NULL, plot = TRUE,
  plotLogRich = FALSE, drop.ZLB = TRUE, timelims = NULL)
}
\arguments{
\item{timeData}{Two-column matrix giving the per-taxon first and last
appearances in absolute time. The matrices output by simFossilTaxa can also
be supplied to taxicDivCont.}

\item{int.length}{The length of intervals used to make the diversity curve.
Ignored if int.times is given.}

\item{int.times}{An optional two-column matrix of the interval start and end
times for calculating the diversity curve. If NULL, calculated internally.
If given, the argument split.int and int.length are ignored.}

\item{plot}{If TRUE, a diversity curve generated from the data is plotted.}

\item{plotLogRich}{If TRUE, taxic diversity is plotted on log scale.}

\item{timelims}{Limits for the x (time) axis for diversity curve plots. Only
affects plotting. Given as either NULL (the default) or as a vector of
length two as for 'xlim' in the basic R function plot. Time axes will be plotted
\emph{exactly} to these values.}

\item{drop.cryptic}{If TRUE, cryptic taxa are merged to form one taxon for
estimating taxon curves. Only works for objects from simFossilTaxa.}

\item{timeList}{A list composed of two matrices, giving interval start and end
dates and taxon first and last occurrences within those intervals. See details.}

\item{drop.singletons}{If TRUE, taxa confined to a single interval will be
dropped prior to the diversity curve calculation. This is sometimes done if
single intervals have overly high diversities due to the 'monograph' effect
where more named taxa are known in certain intervals largely due to
taxonomic expert effort and not real changes in historical biotic diversity.}

\item{extant.adjust}{Amount of time to be added to extend start time for
(0,0) bins for extant taxa, so that the that 'time interval' doesn't appear
to have an infinitely small width.}

\item{split.int}{For discrete time data, should calculated/input intervals
be split at discrete time interval boundaries? If FALSE, can create apparent
artifacts in calculating the diversity curve. See below.}

\item{tree}{A time-scaled phylogeny of class phylo.}

\item{drop.ZLB}{If true, zero-length terminal branches are dropped from the
input tree for phylogenetic datasets, before calculating standing diversity.}
}
\value{
These functions will invisibly return a three-column matrix, where
the first two columns are interval start and end times and the third column
is the number of taxa/lineages counted in that interval.
}
\description{
Functions to plot diversity curves based on taxic range data, in both
discrete and continuous time, and for phylogenies.
}
\details{
First, some background. Diversity curves are plots of species/taxon/lineage richness
over time for a particular group of organisms. For paleontological studies,
these are generally based on per-taxon range data while more recently in
evolutionary biology, molecular phylogenies have been used to calculate
lineage-through-time plots (LTTs). Neither of these approaches are without
their particular weaknesses; reconstructing the true history of biodiversity
is a difficult task no matter what data is available.

The diversity curves produced by these functions will always measure
diversity within binned time intervals (and plot them as rectangular bins).
For continuous-time data or phylogenies, one could decrease the int.length
used to get what is essentially an 'instantaneous' estimate of diversity.
This is warned against, however, as most historical diversity data will have
some time-averaging or uncertain temporal resolution and thus is probably
not finely-resolved enough to calculate instantaneous estimates of
diversity.

As with many functions in the paleotree library, absolute time is always
decreasing, i.e. the present day is zero.

As diversity is counted within binned intervals, the true standing diversity
may be somewhat lower than the measured (observed) quantity, particularly if
intervals are longer than the mean duration of taxa is used. This will be an
issue with all diversity curve functions, but particularly the discrete-time
variant. For diversity data in particularly large discrete time intervals,
plotting this data in smaller bins which do not line up completely with the
original intervals will create a 'spiky' diversity curve, as these smaller
intersecting bins will have a large number of taxa which may have been
present in either of the neighboring intervals. This will give these small
bins an apparently high estimated standing diversity. This artifact is
avoided with the default setting split.int=TRUE, which will split any input
or calculated intervals so that they start and end at the boundaries of the
discrete-time range bins.

The timeList object should be a list composed of two matrices, the first
matrix giving by-interval start and end times (in absolute time), the second
matrix giving the by-taxon first and last appearances in the intervals
defined in the first matrix, numbered as the rows. Absolute time should be
decreasing, while the intervals should be numbered so that the number
increases with time. Taxa alive in the modern should be listed as last
occurring in a time interval that  begins at time 0 and ends at time 0.
See the documentation for the time-scaling  function
\code{\link{bin_timePaleoPhy}} and the simulation function
\code{\link{binTimeData}} for more information on formatting.

Unlike some paleotree functions, such as  perCapitaRates, the intervals
can be overlapping or of unequal length. The diversity curve functions
deal with such issues by assuming taxa occur from the base of the interval
they are first found in until the end of the last interval they are occur
in. Taxa in wide-ranging intervals that contain many others will be treated
as occurring in all nested intervals.

phyloDiv will resolve polytomies to be dichotomous nodes separated by
zero-length branches prior to calculating the diversity curve. There is no
option to alter this behavior, but it should not affect the use of the
function because the addition of the zero-length branches should produce an
identical diversity history as a polytomy. phyloDiv will also drop
zero-length terminal branches, as with the function dropZLB. This the
default behavior for the function but can be turned off by setting the
argument drop.zlb to FALSE.
}
\examples{
#taxicDivDisc with the retiolinae dataset
data(retiolitinae)
taxicDivDisc(retioRanges)

#simulation examples
set.seed(444)
taxa <- simFossilTaxa(p=0.1,q=0.1,nruns=1,mintaxa=20,maxtaxa=30,maxtime=1000,maxExtant=0)
#let's see what the 'true' diversity curve looks like in this case
#plot the FADs and LADs with taxicDivCont()
taxicDivCont(taxa)
#simulate a fossil record with imperfect sampling with sampleRanges
rangesCont <- sampleRanges(taxa,r=0.5)
#plot the diversity curve based on the sampled ranges
layout(1:2)
taxicDivCont(rangesCont)
#Now let's use binTimeData to bin in intervals of 1 time unit
rangesDisc <- binTimeData(rangesCont,int.length=1)
#plot with taxicDivDisc
taxicDivDisc(rangesDisc)
#compare to the continuous time diversity curve

layout(1)
#Now let's make a tree using taxa2phylo
tree <- taxa2phylo(taxa,obs_time=rangesCont[,2])
phyloDiv(tree)

#a simple example with phyloDiv
  #using a tree from rtree in ape
set.seed(444)
tree <- rtree(100)
phyloDiv(tree)

#a neat example of using phyDiv with timeSliceTree
 #to simulate doing molecular-phylogeny studies
 #of diversification...in the past
set.seed(444)
taxa <- simFossilTaxa(p=0.1,q=0.1,nruns=1,mintaxa=20,maxtaxa=30,maxtime=1000,maxExtant=0)
taxicDivCont(taxa)
#that's the whole diversity curve
#with timeSliceTree we could look at the lineage accumulation curve
 #we'd get of species sampled at a point in time
tree <- taxa2phylo(taxa)
#use timeSliceTree to make tree of relationships up until time=950
tree950 <- timeSliceTree(tree,sliceTime=950,plot=TRUE,drop.extinct=FALSE)
#use drop.extinct=T to only get the tree of lineages extant at time=950
tree950 <- timeSliceTree(tree,sliceTime=950,plot=TRUE,drop.extinct=TRUE)
#now its an ultrametric tree with many fewer tips...
#lets plot the lineage accumulation plot on a log scale
phyloDiv(tree950,plotLogRich=TRUE)

#an example of a 'spiky' diversity curve and why split.int is a good thing
set.seed(444)
taxa <- simFossilTaxa(p=0.1,q=0.1,nruns=1,mintaxa=20,maxtaxa=30,maxtime=1000,maxExtant=0)
taxaDiv <- taxicDivCont(taxa)
#simulate a fossil record with imperfect sampling with sampleRanges()
rangesCont <- sampleRanges(taxa,r=0.5)
rangesDisc <- binTimeData(rangesCont,int.length=10)
#now let's plot with taxicDivDisc() but with the intervals from taxaDiv
 #by default, split.int=TRUE
taxicDivDisc(rangesDisc,int.times=taxaDiv[,1:2],split.int=TRUE)
#look pretty
#now let's turn off split.int
taxicDivDisc(rangesDisc,int.times=taxaDiv[,1:2],split.int=FALSE)
#looks 'spiky'!
}
\author{
David W. Bapst
}
\seealso{
\code{\link{multiDiv}}, \code{\link{timeSliceTree}},
\code{\link{binTimeData}}

There are several different functions for traditional LTT plots
(phylogenetic diversity curves), such as the function
,\code{\link{ltt.plot}} in the package ape, the function \code{ltt} in the
package phytools, the function \code{plotLtt} in the package laser and the
function \code{LTT.average.root} in the package TreeSim.
}

