% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/to_tune.R
\name{to_tune}
\alias{to_tune}
\alias{TuneToken}
\title{Indicate that a Parameter Value should be Tuned}
\usage{
to_tune(...)
}
\arguments{
\item{...}{if given, restricts the range to be tuning over, as described above.}
}
\value{
A \code{TuneToken} object.
}
\description{
\code{to_tune()} creates a \code{TuneToken} object which can be assigned to the \verb{$values} slot of a \code{\link{ParamSet}} as an
alternative to a concrete value. This indicates that the value is not given directly but should be tuned using
\CRANpkg{bbotk} or \CRANpkg{mlr3tuning}. If the thus parameterized object
is invoked directly, without being wrapped by or given to a tuner, it will give an error.

The tuning range \code{\link{ParamSet}} that is constructed from the \code{TuneToken} values in a \code{\link{ParamSet}}'s \verb{$values} slot
can be accessed through the \code{ParamSet$search_space()} method. This is done automatically by tuners if no tuning range
is given, but it is also possible to access the \verb{$search_space()} method, modify it further, and give the modified
\code{\link{ParamSet}} to a tuning function (or do anything else with it, nobody is judging you).

A \code{TuneToken} represents the range over which the parameter whose \verb{$values} slot it occupies should be tuned over. It
can be constructed via the \code{to_tune()} function in one of several ways:
\itemize{
\item \strong{\code{to_tune()}}: Indicates a parameter should be tuned over its entire range. Only applies to finite parameters
(i.e. discrete or bounded numeric parameters)
\item \strong{\code{to_tune(lower, upper, logscale)}}: Indicates a numeric parameter should be tuned in the inclusive interval spanning
\code{lower} to \code{upper}, possibly on a log scale if \code{logscale} is se to \code{TRUE}. All parameters are optional, and the
parameter's own lower / upper bounds are used without log scale, by default. Depending on the parameter,
integer (if it is a \code{\link{ParamInt}}) or real values (if it is a \code{\link{ParamDbl}}) are used.\cr
\code{lower}, \code{upper}, and \code{logscale} can be given by position, except when only one of them is given, in which case
it must be named to disambiguate from the following cases.\cr
When \code{logscale} is \code{TRUE}, then a \code{trafo} is generated automatically that transforms to the given bounds. The
bounds are log()'d pre-trafo (see examples). See the \code{logscale} argument of \code{\link{Domain}} functions for more info.\cr
Note that "logscale" is \emph{not} inherited from the \code{\link{Param}} that the \code{TuneToken} belongs to! Defining a parameter
with \verb{p_dbl(... logscale = TRUE)} will \emph{not} automatically give the \code{to_tune()} assigned to it log-scale.
\item \strong{\code{to_tune(levels)}}: Indicates a parameter should be tuned through the given discrete values. \code{levels} can be any
named or unnamed atomic vector or list (although in the unnamed case it must be possible to construct a
corresponding \code{character} vector with distinct values using \code{as.character}).
\item \strong{\verb{to_tune(<Domain>)}}: The given \code{\link{Domain}} object (constructed e.g. with \code{\link[=p_int]{p_int()}} or \code{\link[=p_fct]{p_fct()}}) indicates
the range which should be tuned over. The supplied \code{trafo} function is used for parameter transformation.
\item \strong{\verb{to_tune(<Param>)}}: The given \code{\link{Param}} object indicates the range which should be tuned over.
\item \strong{\verb{to_tune(<ParamSet>)}}: The given \code{\link{ParamSet}} is used to tune over a single \code{Param}. This is useful for cases
where a single evaluation-time parameter value (e.g. \code{\link{ParamUty}}) is constructed from multiple tuner-visible
parameters (which may not be \code{ParamUty}). The supplied \code{\link{ParamSet}} should always contain a \verb{$trafo} function,
which must always return a \code{list} with a single entry.
}

The \code{TuneToken} object's internals are subject to change and should not be relied upon. \code{TuneToken} objects should
only be constructed via \code{to_tune()}, and should only be used by giving them to \verb{$values} of a \code{\link{ParamSet}}.
}
\examples{
params = ParamSet$new(list(
  ParamInt$new("int", 0, 10),
  ParamInt$new("int_unbounded"),
  ParamDbl$new("dbl", 0, 10),
  ParamDbl$new("dbl_unbounded"),
  ParamDbl$new("dbl_bounded_below", lower = 1),
  ParamFct$new("fct", c("a", "b", "c")),
  ParamUty$new("uty1"),
  ParamUty$new("uty2"),
  ParamUty$new("uty3"),
  ParamUty$new("uty4"),
  ParamUty$new("uty5")
))

params$values = list(

  # tune over entire range of `int`, 0..10:
  int = to_tune(),

  # tune over 2..7:
  int_unbounded = to_tune(2, 7),

  # tune on a log scale in range 1..10;
  # recognize upper bound of 10 automatically, but restrict lower bound to 1:
  dbl = to_tune(lower = 1, logscale = TRUE),
  ## This is equivalent to the following:
  # dbl = to_tune(p_dbl(log(1), log(10), trafo = exp)),

  # nothing keeps us from tuning a dbl over integer values
  dbl_unbounded = to_tune(p_int(1, 10)),

  # tune over values "a" and "b" only
  fct = to_tune(c("a", "b")),

  # tune over integers 2..8.
  # ParamUty needs type information in form of p_xxx() in to_tune.
  uty1 = to_tune(p_int(2, 8)),

  # tune uty2 like a factor, trying 1, 10, and 100:
  uty2 = to_tune(c(1, 10, 100)),

  # tune uty3 like a factor. The factor levels are the names of the list
  # ("exp", "square"), but the trafo will generate the values from the list.
  # This way you can tune an objective that has function-valued inputs.
  uty3 = to_tune(list(exp = exp, square = function(x) x^2)),

  # tune through multiple parameters. When doing this, the ParamSet in tune()
  # must have the trafo that generates a list with one element and the right
  # name:
  uty4 = to_tune(ps(
    base = p_dbl(0, 1),
    exp = p_int(0, 3),
    .extra_trafo = function(x, param_set) {
      list(uty4 = x$base ^ x$exp)
    }
  )),

  # not all values need to be tuned!
  uty5 = 100
)

print(params$values)

print(params$search_space())

# Change `$values` directly and generate new `$search_space()` to play around
params$values$uty3 = 8
params$values$uty2 = to_tune(c(2, 4, 8))

print(params$search_space())

# Notice how `logscale` applies `log()` to lower and upper bound pre-trafo:
params = ParamSet$new(list(ParamDbl$new("x")))

params$values$x = to_tune(1, 100, logscale = TRUE)

print(params$search_space())

grid = generate_design_grid(params$search_space(), 3)

# The grid is equidistant within log-bounds pre-trafo:
print(grid)

# But the values are on a log scale scale with desired bounds after trafo:
print(grid$transpose())

}
\seealso{
Other ParamSet construction helpers: 
\code{\link{Domain}},
\code{\link{ps}()}
}
\concept{ParamSet construction helpers}
