% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wue.R
\name{pwue}
\alias{pwue}
\title{Calculate pseudo water use efficiency from phenotype and watering data}
\usage{
pwue(
  df,
  w = NULL,
  pheno = "area_pixels",
  time = "timestamp",
  id = "barcode",
  offset = 0,
  pre_watering = "weight_before",
  post_watering = "weight_after",
  method = "rate"
)
}
\arguments{
\item{df}{Dataframe containing wide single-value phenotype data.
This should already be aggregated to one row per plant per day (angles/rotations combined).}

\item{w}{Watering data as returned from bw.water.}

\item{pheno}{Phenotype column name, defaults to "area_pixels"}

\item{time}{Variable(s) that identify a plant on a given day.
Defaults to \code{c("barcode", "DAS")}.}

\item{id}{Variable(s) that identify a plant over time. Defaults to \code{"barcode"}.}

\item{offset}{Optionally you can specify how long before imaging a watering should not be taken into
account. This defaults to 0, meaning that if a plant were watered directly before being imaged then
that water would be counted towards WUE between the current image and the prior one.
This argument is taken to be in seconds.}

\item{pre_watering}{Column containing weight before watering in \code{w},
defaults to "weight_before".}

\item{post_watering}{Column containing weight after watering in \code{w},
defaults to "weight_after".}

\item{method}{Which method to use, options are "rate", "abs", and "ndt". The "rate" method considers
WUE as the change in a phenotype divided by the amount of water added. The "abs" method considers WUE
as the amount of water used by a plant given its absolute size.
The "ndt" method calculates normalized daily transpiration,
which is the reciprocal of the "abs" method. The "rate" method is for questions more
related to efficiency in using water to grow while "abs"/"ndt" are more suited to questions about
how efficient a plant is at maintaining size given some amount of water or how much water it uses
at a given size.}
}
\value{
A data frame containing the watering data and
    to phenotype data with new columns for change in the phenotype (\code{pheno_diff}),
    amount of water used (\code{total_water}) over the interval between phenotype measurements
    (water added post to pre phenotype measurement), \code{start} and \code{end} times for the
    interval as well as their difference (\code{timeLengthSeconds}), and pseudo water use efficiency
    (\code{pWUE}).
}
\description{
Rate based water use efficiency (WUE) is the change in biomass per unit of water
metabolized. Using image based phenotypes and watering data we can calculate pseudo-WUE (pwue) over
time. Here area_pixels is used as a proxy for biomass and transpiration is approximated using
watering data. The equation is then
\eqn{
\frac{P_{t} - P_{t-1}}{W_{t_{end-1}}-W_{t_{start}} }}{P_[t] - P_[t-1] / W_[t_(end-1)]-W_[t_start]
},
where P is the phenotype and W is the weight before watering.

Absolute value based WUE is the amount of water used to sustain a plants biomass over a given period.
The equation is then
\eqn{\frac{P_{t}}{W_{t_{end-1}}-W_{t_{start}} }}{P_[t] / W_[t_(end-1)]-W_[t_start]}
}
\examples{

set.seed(123)
weight_before <- sort(round(rnorm(20, 100, 10), 0))
weight_after <- sort(round(rnorm(20, 120, 10), 0))
df <- data.frame(
  time = seq_along(weight_before),
  area_pixels = round(130 / (1 + exp( (12 - seq_along(weight_before))/3) ), 0),
  weight_before,
  weight_after,
  barcode = 1,
  other = "x"
)
ex <- pwue(df, time = "time", method = "rate", id = c("barcode", "other"))
w <- df[, c("time", "weight_before", "weight_after", "barcode")]
ex2 <- pwue(df, w, time = c("time", "time"), method = "abs")
ex3 <- pwue(df, w, time = c("time", "time"), method = "ndt")

}
\keyword{WUE}
