#' Add (Surplus and Gross) Domestic Product Data
#'
#' @description
#'
#' \code{add_sdp_gdp()} allows you to add estimated GDP and "surplus"
#' domestic product data from a 2020 analysis published in
#' *International Studies Quarterly* by Anders, Fariss, and Markowitz.
#'
#'
#' @return
#'
#' \code{add_sdp_gdp()} takes a (dyad-year, leader-year, leader-dyad-year,
#' state-year)  data frame and adds information about the estimated gross
#' domestic product (in 2011 USD) for that year, the estimated population
#' in that year, the GDP per capita in that year, and what Anders, Fariss
#' and Markowitz term the "surplus domestic product" in that year. If the
#' data are dyad-year (leader-dyad-year), the function adds eight total
#' columns for the first state (i.e. *ccode1*) and the second state
#' (i.e. *ccode2*) for all these estimates. If the data are state-year
#' (or leader-year), the function returns four additional columns to the
#' original data that contain that same information for a given state in
#' a given year.
#'
#' @details
#'
#' The function leans on attributes of the data that are provided by one of
#' the "create" functions. Make sure a recognized function (or data created
#' by that function) appear at the top of the proverbial pipe. Users will
#' also want to note that the underlying function access two different data
#' sets. It appears that the results published in the
#' *International Studies Quarterly* used Correlates of War classification,
#' but a follow-up repository on Github uses Gleditsch-Ward classification.
#' The extent to which these estimates are generated by simulation, it does
#' mean the estimates will be slightly different across both data sets even
#' for common observations (e.g. the United States in 1816).
#'
#' Because these are large nominal numbers, the estimates have been
#' log-transformed. Users can always exponentiate these if they choose.
#' Researchers can use these data to construct reasonable estimates of
#' surplus GDP per capita, but must exponentiate the underlying variables
#' before doing this.
#'
#' Be mindful that the data are fundamentally state-year and that extensions
#' to leader-level data should be understood as approximations for leaders in
#' a given state-year.
#'
#'
#' @author Steven V. Miller
#'
#' @param data a data frame with appropriate \pkg{peacesciencer} attributes
#'
#' @references
#'
#' Anders, Therese, Christopher J. Fariss, and Jonathan N. Markowitz. 2020.
#' "Bread Before Guns or Butter: Introducing Surplus Domestic Product (SDP)"
#' \emph{International Studies Quarterly} 64(2): 392--405.
#'
#'
#' @examples
#'
#' # just call `library(tidyverse)` at the top of the your script
#' library(magrittr)
#'
#' cow_ddy %>% add_sdp_gdp()
#'
#' create_stateyears() %>% add_sdp_gdp()
#'
#' create_stateyears(system = "gw") %>% add_sdp_gdp()
#'
#'
add_sdp_gdp <- function(data) {

  if (length(attributes(data)$ps_data_type) > 0 && attributes(data)$ps_data_type %in% c('dyad_year', 'leader_dyad_year')) {

    if (length(attributes(data)$ps_system) > 0 && attributes(data)$ps_system == "cow") {

      data %>%
        left_join(., cow_sdp_gdp, by=c("ccode1"="ccode","year"="year")) %>%
        rename(wbgdp2011est1 = .data$wbgdp2011est,
               wbpopest1 = .data$wbpopest,
               wbgdppc2011est1 = .data$wbgdppc2011est,
               sdpest1 = .data$sdpest) %>%
        left_join(., cow_sdp_gdp, by=c("ccode2"="ccode","year"="year")) %>%
        rename(wbgdp2011est2 = .data$wbgdp2011est,
               wbpopest2 = .data$wbpopest,
               wbgdppc2011est2 = .data$wbgdppc2011est,
               sdpest2 = .data$sdpest) -> data

      return(data)

    } else if (length(attributes(data)$ps_system) > 0 && attributes(data)$ps_system == "gw") {

      data %>%
        left_join(., gw_sdp_gdp, by=c("gwcode1"="gwcode","year"="year")) %>%
        rename(wbgdp2011est1 = .data$wbgdp2011est,
               wbpopest1 = .data$wbpopest,
               wbgdppc2011est1 = .data$wbgdppc2011est,
               sdpest1 = .data$sdpest) %>%
        left_join(., gw_sdp_gdp, by=c("gwcode2"="gwcode","year"="year")) %>%
        rename(wbgdp2011est2 = .data$wbgdp2011est,
               wbpopest2 = .data$wbpopest,
               wbgdppc2011est2 = .data$wbgdppc2011est,
               sdpest2 = .data$sdpest) -> data


      return(data)

    } else {

      stop("add_sdp_gdp() requires either Correlates of War ('cow') or Gleditsch-Ward ('gw') as system type.")
    }

  } else if (length(attributes(data)$ps_data_type) > 0 && attributes(data)$ps_data_type %in% c("state_year", "leader_year")) {

    if (length(attributes(data)$ps_system) > 0 && attributes(data)$ps_system == "cow") {

      cow_sdp_gdp %>%
        left_join(data, .) -> data

      return(data)

    } else if (length(attributes(data)$ps_system) > 0 && attributes(data)$ps_system == "gw") {

      gw_sdp_gdp %>%
        left_join(data, .) -> data

      return(data)

    } else {

      stop("add_sdp_gdp() requires either Correlates of War ('cow') or Gleditsch-Ward ('gw') as system type.")
    }

  } else {

    stop("add_sdp_gdp() requires a data/tibble with attributes$ps_data_type of state_year or dyad_year. Try running create_dyadyears() or create_stateyears() at the start of the pipe.")

  }

}



