% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_prclmm.R
\name{fit_prclmm}
\alias{fit_prclmm}
\title{Step 3 of PRC-LMM (estimation of the penalized Cox model(s))}
\usage{
fit_prclmm(object, surv.data, baseline.covs = NULL, penalty = "ridge",
  standardize = TRUE, pfac.base.covs = 0, n.alpha.elnet = 11,
  n.folds.elnet = 5, n.cores = 1, verbose = TRUE)
}
\arguments{
\item{object}{the output of step 2 of the PRC-LMM procedure, 
as produced by the \code{\link{summarize_lmms}} function}

\item{surv.data}{a data frame with the survival data and (if 
relevant) additional baseline covariates. \code{surv.data} should at least
contain a subject id (called \code{id}), the time to event outcome  
(\code{time}), and binary event variable (\code{event})}

\item{baseline.covs}{a formula specifying the variables 
(e.g., baseline age) in \code{surv.data} that should be included 
as baseline covariates in the penalized Cox model. Example:
\code{baseline.covs = '~ baseline.age'}. Default is \code{NULL}}

\item{penalty}{the type of penalty function used for regularization.
Default is \code{'ridge'}, other possible values are \code{'elasticnet'} 
and \code{'lasso'}}

\item{standardize}{logical argument: should the predicted random effects
be standardized when included in the penalized Cox model? Default is \code{TRUE}}

\item{pfac.base.covs}{a single value, or a vector of values, indicating
whether the baseline covariates (if any) should be penalized (1) or not (0).
Default is \code{pfac.base.covs = 0} (no penalization of all baseline covariates)}

\item{n.alpha.elnet}{number of alpha values for the two-dimensional 
grid of tuning parameteres in elasticnet.
Only relevant if \code{penalty = 'elasticnet'}. Default is 11,
so that the resulting alpha grid is c(1, 0.95, 0.90, ..., 0.05, 0)}

\item{n.folds.elnet}{number of folds to be used for the selection
of the tuning parameter in elasticnet. Only relevant if 
\code{penalty = 'elasticnet'}. Default is 5}

\item{n.cores}{number of cores to use to parallelize the computation
of the cluster bootstrap optimism correction procedure. If 
\code{ncores = 1} (default), no parallelization is done. 
Pro tip: you can use \code{parallel::detectCores()} to check 
how many cores are available on your computer}

\item{verbose}{if \code{TRUE} (default and recommended value), information
on the ongoing computations is printed in the console}
}
\value{
A list containing the following objects:
\itemize{
\item \code{call}: the function call
\item \code{pcox.orig}: the penalized Cox model fitted on the
original dataset;
\item \code{surv.data}: the supplied survival data (ordered by
subject id)
\item \code{n.boots}: number of bootstrap samples;
\item \code{boot.ids}: a list with the ids of bootstrapped subjects 
(when \code{n.boots > 0});
\item \code{pcox.boot}: a list where each element is a fitted penalized
Cox model for a given bootstrap sample (when \code{n.boots > 0}).
}
}
\description{
This function performs the third step for the estimation
of the PRC-LMM model proposed in Signorelli et al. (2020, 
in review)
}
\examples{
# generate example data
set.seed(1234)
p = 4 # number of longitudinal predictors
simdata = simulate_prclmm_data(n = 100, p = p, p.relev = 2, 
             seed = 123, t.values = c(0, 0.2, 0.5, 1, 1.5, 2))
             
# specify options for cluster bootstrap optimism correction
# procedure and for parallel computing 
do.bootstrap = FALSE
# IMPORTANT: set do.bootstrap = TRUE to compute the optimism correction!
n.boots = ifelse(do.bootstrap, 100, 0)
parallelize = FALSE
# IMPORTANT: set parallelize = TRUE to speed computations up!
if (!parallelize) n.cores = 1
if (parallelize) {
   # identify number of available cores on your machine
   n.cores = parallel::detectCores()
   if (is.na(n.cores)) n.cores = 1
}

# step 1 of PRC-LMM: estimate the LMMs
y.names = paste('marker', 1:p, sep = '')
step1 = fit_lmms(y.names = y.names, 
                 fixefs = ~ age, ranefs = ~ age | id, 
                 long.data = simdata$long.data, 
                 surv.data = simdata$surv.data,
                 t.from.base = t.from.base,
                 n.boots = n.boots, n.cores = n.cores)
                 
# step 2 of PRC-LMM: compute the summaries 
# of the longitudinal outcomes
step2 = summarize_lmms(object = step1, n.cores = n.cores)

# step 3 of PRC-LMM: fit the penalized Cox models
step3 = fit_prclmm(object = step2, surv.data = simdata$surv.data,
                   baseline.covs = ~ baseline.age,
                   penalty = 'ridge', n.cores = n.cores)
}
\references{
Signorelli, M., Spitali, P., Al-Khalili Szigyarto, C, 
The MARK-MD Consortium, Tsonaka, R. (2021). 
Penalized regression calibration: a method for the prediction 
of survival outcomes using complex longitudinal and 
high-dimensional data. arXiv preprint: arXiv:2101.04426.
}
\seealso{
\code{\link{fit_lmms}} (step 1), 
\code{\link{summarize_lmms}} (step 2),
\code{\link{performance_prclmm}}
}
\author{
Mirko Signorelli
}
