% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/performance_pencox_baseline.R
\name{performance_pencox_baseline}
\alias{performance_pencox_baseline}
\title{Predictive performance of the penalized Cox model
with baseline covariates}
\usage{
performance_pencox_baseline(fitted_pencox, times = 1, n.cores = 1,
  verbose = TRUE)
}
\arguments{
\item{fitted_pencox}{the output of \code{\link{pencox_baseline}}}

\item{times}{numeric vector with the time points at which
to estimate the time-dependent AUC}

\item{n.cores}{number of cores to use to parallelize the computation
of the CBOCP procedure. If \code{ncores = 1} (default), 
no parallelization is done. Pro tip: you can use 
\code{parallel::detectCores()} to check how many 
cores are available on your computer}

\item{verbose}{if \code{TRUE} (default and recommended value), information
on the ongoing computations is printed in the console}
}
\value{
A list containing the following objects:
\itemize{
\item \code{call}: the function call;
\item \code{concordance}: a data frame with the naive and
optimism-corrected estimates of the concordance (C) index;
\item \code{tdAUC}: a data frame with the naive and
optimism-corrected estimates of the time-dependent AUC
at the desired time points.
}
}
\description{
This function computes the naive and optimism-corrected
measures of performance (C index and time-dependent AUC) 
for a penalized Cox model with baseline covariates as
presented in Signorelli et al. (2021). 
The optimism correction is a bootstrap
optimism correction procedure
}
\examples{
# generate example data
set.seed(1234)
p = 4 # number of longitudinal predictors
simdata = simulate_prclmm_data(n = 100, p = p, p.relev = 2, 
             seed = 123, t.values = c(0, 0.2, 0.5, 1, 1.5, 2))
# create dataframe with baseline measurements only
baseline.visits = simdata$long.data[which(!duplicated(simdata$long.data$id)),]
df = cbind(simdata$surv.data, baseline.visits)
df = df[ , -c(5:7)]

do.bootstrap = FALSE
# IMPORTANT: set do.bootstrap = TRUE to compute the optimism correction!
n.boots = ifelse(do.bootstrap, 100, 0)
more.cores = FALSE
# IMPORTANT: set more.cores = TRUE to speed computations up!
if (!more.cores) n.cores = 2
if (more.cores) {
   # identify number of available cores on your machine
   n.cores = parallel::detectCores()
   if (is.na(n.cores)) n.cores = 2
}

form = as.formula(~ baseline.age + marker1 + marker2
                     + marker3 + marker4)
base.pcox = pencox_baseline(data = df, 
              formula = form, 
              n.boots = n.boots, n.cores = n.cores) 
ls(base.pcox)
                   
# compute the performance measures
perf = performance_pencox_baseline(fitted_pencox = base.pcox, 
          times = c(0.5, 1, 1.5, 2), n.cores = n.cores)

# concordance index:
perf$concordance
# time-dependent AUC:
perf$tdAUC
}
\references{
Signorelli, M., Spitali, P., Al-Khalili Szigyarto, C, 
The MARK-MD Consortium, Tsonaka, R. (2021). 
Penalized regression calibration: a method for the prediction 
of survival outcomes using complex longitudinal and 
high-dimensional data. Statistics in Medicine, 40 (27), 6178-6196.
DOI: 10.1002/sim.9178
}
\seealso{
\code{\link{pencox_baseline}}
}
\author{
Mirko Signorelli
}
