#' @title Binned residuals for logistic regression
#' @name binned_residuals
#'
#' @description Check model quality of logistic regression models.
#'
#' @param model A \code{glm}-object with binomial-family.
#' @param term Name of independent variable from \code{x}. If not \code{NULL},
#'   average residuals for the categories of \code{term} are plotted; else,
#'   average residuals for the estimated probabilities of the response are
#'   plotted.
#' @param n_nins Numeric, the number of bins to divide the data. If
#'   \code{n_nins = NULL}, the square root of the number of observations is
#'   taken.
#'
#' @return A data frame representing the data that is mapped to the plot, which is
#'   automatically plotted. In case all residuals are inside the error bounds,
#'   points are black. If some of the residuals are outside the error bounds
#'   (indicates by the grey-shaded area), blue points indicate residuals that
#'   are OK, while red points indicate model under- or overfitting for the
#'   related range of estimated probabilities.
#'
#' @details Binned residual plots are achieved by \dQuote{dividing the data into
#'   categories (bins) based on their fitted values, and then plotting
#'   the average residual versus the average fitted value for each bin.}
#'   \cite{(Gelman, Hill 2007: 97)}. If the model were true, one would
#'   expect about 95\% of the residuals to fall inside the error bounds.
#'   \cr \cr
#'   If \code{term} is not \code{NULL}, one can compare the residuals in
#'   relation to a specific model predictor. This may be helpful to check
#'   if a term would fit better when transformed, e.g. a rising and falling
#'   pattern of residuals along the x-axis (the pattern is indicated by
#'   a green line) is a signal to consider taking the logarithm of the
#'   predictor (cf. Gelman and Hill 2007, pp. 97ff).
#'
#' @references Gelman, A., & Hill, J. (2007). Data analysis using regression and multilevel/hierarchical models. Cambridge; New York: Cambridge University Press.
#'
#' @examples
#' model <- glm(vs ~ wt + mpg, data = mtcars, family = "binomial")
#' binned_residuals(model)
#'
#' @importFrom stats fitted sd complete.cases
#' @importFrom insight get_data get_response find_response
#' @export
binned_residuals <- function(model, term = NULL, n_nins = NULL) {
  fv <- stats::fitted(model)
  mf <- insight::get_data(model)

  if (is.null(term))
    pred <- fv
  else
    pred <- mf[[term]]

  y <- recode_to_zero(as.numeric(as.character(insight::get_response(model)))) - fv

  if (is.null(n_nins)) n_nins <- round(sqrt(length(pred)))

  breaks.index <- floor(length(pred) * (1:(n_nins - 1)) / n_nins)
  breaks <- unique(c(-Inf, sort(pred)[breaks.index], Inf))

  model.binned <- as.numeric(cut(pred, breaks))

  d <- suppressWarnings(lapply(1:n_nins, function(.x) {
      items <- (1:length(pred))[model.binned == .x]
      model.range <- range(pred[items], na.rm = TRUE)
      xbar <- mean(pred[items], na.rm = TRUE)
      ybar <- mean(y[items], na.rm = TRUE)
      n <- length(items)
      sdev <- stats::sd(y[items], na.rm = TRUE)

      data.frame(
        xbar = xbar,
        ybar = ybar,
        n = n,
        x.lo = model.range[1],
        x.hi = model.range[2],
        se = 2 * sdev / sqrt(n)
      )
  }))

  d <- do.call(rbind, d)
  d <- d[stats::complete.cases(d), ]

  gr <- abs(d$ybar) > abs(d$se)
  d$group <- "yes"
  d$group[gr] <- "no"

  class(d) <- c("binned_residuals", class(d))
  attr(d, "resp_var") <- insight::find_response(model)
  attr(d, "term") <- term

  d
}
