% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_outliers.R
\name{check_outliers}
\alias{check_outliers}
\alias{check_outliers.default}
\alias{check_outliers.numeric}
\alias{check_outliers.data.frame}
\title{Outliers detection (check for influential observations)}
\usage{
check_outliers(x, ...)

\method{check_outliers}{default}(x, method = c("cook", "pareto"), threshold = NULL, ...)

\method{check_outliers}{numeric}(x, method = "zscore", threshold = NULL, ...)

\method{check_outliers}{data.frame}(x, method = "mahalanobis", threshold = NULL, ...)
}
\arguments{
\item{x}{A model or a data.frame object.}

\item{...}{When \code{method = "ics"}, further arguments in \code{...} are
passed down to \code{ICSOutlier::ics.outlier()}.}

\item{method}{The outlier detection method(s). Can be "all" or some of c("cook", "pareto", "zscore", "iqr", "mahalanobis", "robust", "mcd", "ics", "optics", "lof").}

\item{threshold}{A list containing the threshold values for each method (e.g. \code{list('mahalanobis' = 7, 'cook' = 1)}), above which an observation is
considered as outlier. If \code{NULL}, default values will be used (see 'Details').}
}
\value{
A logical vector of the detected outliers with a nice printing
  method: a check (message) on whether outliers were detected or not. The
  information on the distance measure and whether or not an observation is
  considered as outlier can be recovered with the \code{as.data.frame}
  function.
}
\description{
Checks for and locates influential observations (i.e., "outliers") via several distance and/or clustering methods. If several methods are selected, the returned "Outlier" vector will be a composite outlier score, made of the average of the binary (0 or 1) results of each method. It represents the probability of each observation of being classified as an outlier by at least one method. The decision rule used by default is to classify as outliers observations which composite outlier score is superior or equal to 0.5 (i.e., that were classified as outliers by at least half of the methods). See the \strong{Details} section below for a description of the methods.
}
\details{
Outliers can be defined as particularly influential observations. Most methods rely on the computation of some distance metric, and the observations greater than a certain threshold are considered outliers. Importantly, outliers detection methods are meant to provide information to the researcher, rather than being an automatized procedure which mindless application is a substitute for thinking.

An \strong{example sentence} for reporting the usage of the composite method could be:

\emph{"Based on a composite outlier score (see the 'check_outliers' function in the 'performance' R package; Lüdecke et al., 2019) obtained via the joint application of multiple outliers detection algorithms (Z-scores, Iglewicz, 1993; Interquartile range (IQR); Mahalanobis distance, Cabana, 2019; Robust Mahalanobis distance, Gnanadesikan & Kettenring, 1972; Minimum Covariance Determinant, Leys et al., 2018; Invariant Coordinate Selection, Archimbaud et al., 2018; OPTICS, Ankerst et al., 1999; Isolation Forest, Liu et al. 2008; and Local Outlier Factor, Breunig et al., 2000), we excluded n participants that were classified as outliers by at least half of the methods used."}

\subsection{Model-specific methods}{
\itemize{
\item \strong{Cook's Distance}:
 Among outlier detection methods, Cook's distance and leverage are less common
 than the basic Mahalanobis distance, but still used. Cook's distance estimates
 the variations in regression coefficients after removing each observation,
 one by one (Cook, 1977). Since Cook's distance is in the metric of an F
 distribution with p and n-p degrees of freedom, the median point of the quantile
 distribution can be used as a cut-off (Bollen, 1985). A common approximation
 or heuristic is to use 4 divided by the numbers of observations, which usually
 corresponds to a lower threshold (i.e., more outliers are detected). This only works for Frequentist models. For Bayesian models, see \code{pareto}.

\item \strong{Pareto}:
The reliability and approximate convergence of Bayesian models can be assessed using the estimates for the shape parameter k of the generalized Pareto distribution. If the estimated tail shape parameter k exceeds 0.5, the user should be warned, although in practice the authors of the \code{loo} package observed good performance for values of k up to 0.7 (the default threshold used by \code{performance}).
}}


\subsection{Univariate methods}{
\itemize{
\item \strong{Z-scores}:
 The Z-score, or standard score, is a way of describing a data point as deviance from a central value, in terms of standard deviations from the mean or, as it is here the case by default (Iglewicz, 1993), in terms of Median Absolute Deviation (MAD) from the median (which are robust measures of dispersion and centrality). The default threshold to classify outliers is 1.959 (\code{threshold = list("zscore" = = 1.959)}), corresponding to the 2.5\% (\code{qnorm(0.975)}) most extreme observations (assuming the data is normally distributed). Importantly, the Z-score method is univariate: it is computed column by column. If a data.frame is passed, then the maximum distance is kept for each observations. Thus, all observations that are extreme for at least one variable will be detected. However, this method is not suited for high dimensional data (with many columns), returning too liberal results (detecting many outliers).

\item \strong{IQR}:
 Using the IQR (interquartile range) is a robust method developed by John Tukey, which often appears in box-and-whisker plots (e.g., in \code{geom_boxplot}). The interquartile range is the range between the first and the third quartiles. Tukey considered as outliers any data point that fell outside of either 1.5 times (the default threshold) the IQR below the first or above the third quartile. Similar to the Z-score method, this is a univariate method for outliers detection, returning outliers detected for at least one column, and might thus not be suited to high dimensional data.
}}

\subsection{Multivariate methods}{
\itemize{
\item \strong{Mahalanobis Distance}:
Mahalanobis distance (Mahalanobis, 1930) is often used for multivariate outliers
detection as this distance takes into account the shape of the observations.
The default \code{threshold} is often arbitrarily set to some deviation (in
terms of SD or MAD) from the mean (or median) of the Mahalanobis distance.
However, as the Mahalanobis distance can be approximated by a Chi squared
distribution (Rousseeuw & Van Zomeren, 1990), we can use the alpha quantile
of the chi-square distribution with k degrees of freedom (k being the number of
columns). By default, the alpha threshold is set to 0.025 (corresponding to the
2.5\% most extreme observations; Cabana, 2019). This criterion is a natural extension of the
median plus or minus a coefficient times the MAD method (Leys et al., 2013).

\item \strong{Robust Mahalanobis Distance}:
A robust version of Mahalanobis distance using an Orthogonalized
Gnanadesikan-Kettenring pairwise estimator (Gnanadesikan \& Kettenring, 1972).
Requires the \pkg{bigutilsr} package. See the \code{bigutilsr::dist_ogk()} function.

\item \strong{Minimum Covariance Determinant (MCD)}:
Another robust version of Mahalanobis. Leys et al. (2018) argue that Mahalanobis Distance is not a robust way to
determine outliers, as it uses the means and covariances of all the data
– including the outliers – to determine individual difference scores. Minimum
Covariance Determinant calculates the mean and covariance matrix based on the
most central subset of the data (by default, 66\%), before computing the
Mahalanobis Distance. This is deemed to be a more robust method of identifying
and removing outliers than regular Mahalanobis distance.

\item \strong{Invariant Coordinate Selection (ICS)}:
 The outlier are detected using ICS, which by default uses an alpha threshold
 of 0.025 (corresponding to the 2.5\% most extreme observations) as a cut-off value for outliers classification. Refer to the help-file
 of \code{ICSOutlier::ics.outlier()} to get more details about this procedure.
 Note that \code{method = "ics"} requires both \pkg{ICS} and \pkg{ICSOutlier}
 to be installed, and that it takes some time to compute the results.

\item \strong{OPTICS}:
 The Ordering Points To Identify the Clustering Structure (OPTICS) algorithm (Ankerst et al., 1999) is using similar concepts to DBSCAN (an unsupervised clustering technique that can be used for outliers detection). The threshold argument is passed as \code{minPts}, which corresponds to the minimum size of a cluster. By default, this size is set at 2 times the number of columns (Sander et al., 1998). Compared to the others techniques, that will always detect several outliers (as these are usually defined as a percentage of extreme values), this algorithm functions in a different manner and won't always detect outliers. Note that \code{method = "optics"} requires the \pkg{dbscan} package to be installed, and that it takes some time to compute the results.

\item \strong{Isolation Forest}:
 The outliers are detected using the anomaly score of an isolation forest (a class of random forest). The default threshold
 of 0.025 will classify as outliers the observations located at \code{qnorm(1-0.025) * MAD)} (a robust equivalent of SD) of the median (roughly corresponding to the 2.5\% most extreme observations). Requires the \pkg{solitude} package.

 \item \strong{Local Outlier Factor}:
 Based on a K nearest neighbours algorithm, LOF compares the local density
 of an point to the local densities of its neighbors instead of computing a
 distance from the center (Breunig et al., 2000). Points that have a
 substantially lower density than their neighbors are considered outliers.
 A LOF score of approximately 1 indicates that density around the point is
 comparable to its neighbors. Scores significantly larger than 1 indicate outliers.
 The default threshold of 0.025 will classify as outliers the observations
 located at \code{qnorm(1-0.025) * SD)} of the log-transformed LOF distance.
 Requires the \pkg{dbscan} package.
}}

\subsection{Threshold specification}{
Default thresholds are currently specified as follows:

\preformatted{
list(
  zscore = stats::qnorm(p = 1 - 0.025),
  iqr = 1.5,
  cook = stats::qf(0.5, ncol(x), nrow(x) - ncol(x)),
  pareto = 0.7,
  mahalanobis = stats::qchisq(p = 1 - 0.025, df = ncol(x)),
  robust = stats::qchisq(p = 1 - 0.025, df = ncol(x)),
  mcd = stats::qchisq(p = 1 - 0.025, df = ncol(x)),
  ics = 0.025,
  optics = 2 * ncol(x),
  iforest = 0.025,
  lof = 0.025
)
}}
}
\note{
There is also a \href{https://easystats.github.io/see/articles/performance.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}. \strong{Please note} that the range of the distance-values along the y-axis is re-scaled to range from 0 to 1.
}
\examples{
data <- mtcars # Size nrow(data) = 32

# For single variables ------------------------------------------------------
outliers_list <- check_outliers(data$mpg) # Find outliers
outliers_list # Show the row index of the outliers
as.numeric(outliers_list) # The object is a binary vector...
filtered_data <- data[!outliers_list, ] # And can be used to filter a dataframe
nrow(filtered_data) # New size, 28 (4 outliers removed)


# For dataframes ------------------------------------------------------
check_outliers(data) # It works the same way on dataframes

# You can also use multiple methods at once
outliers_list <- check_outliers(data, method = c(
  "mahalanobis",
  "iqr",
  "zscore"
))
outliers_list
# Using `as.data.frame()`, we can access more details!
outliers_info <- as.data.frame(outliers_list)
head(outliers_info)
outliers_info$Outlier # Including the probability of being an outlier
# And we can be more stringent in our outliers removal process
filtered_data <- data[outliers_info$Outlier < 0.1, ]
\dontrun{
# You can also run all the methods
check_outliers(data, method = "all")

# For statistical models ---------------------------------------------
# select only mpg and disp (continuous)
mt1 <- mtcars[, c(1, 3, 4)]
# create some fake outliers and attach outliers to main df
mt2 <- rbind(mt1, data.frame(
  mpg = c(37, 40), disp = c(300, 400),
  hp = c(110, 120)
))
# fit model with outliers
model <- lm(disp ~ mpg + hp, data = mt2)

outliers_list <- check_outliers(model)
# plot(outliers_list)
insight::get_data(model)[outliers_list, ] # Show outliers data

if (require("MASS")) {
  check_outliers(model, method = c("mahalabonis", "mcd"))
}
if (require("ICS")) {
  # This one takes some seconds to finish...
  check_outliers(model, method = "ics")
}
}
}
\references{
\itemize{
\item Archimbaud, A., Nordhausen, K., \& Ruiz-Gazen, A. (2018). ICS for multivariate outlier detection with application to quality control. Computational Statistics & Data Analysis, 128, 184–199. \doi{10.1016/j.csda.2018.06.011}
\item Gnanadesikan, R., \& Kettenring, J. R. (1972). Robust estimates, residuals, and outlier detection with multiresponse data. Biometrics, 81-124.
\item Bollen, K. A., & Jackman, R. W. (1985). Regression diagnostics: An expository treatment of outliers and influential cases. Sociological Methods & Research, 13(4), 510-542.
\item Cabana, E., Lillo, R. E., \& Laniado, H. (2019). Multivariate outlier detection based on a robust Mahalanobis distance with shrinkage estimators. arXiv preprint arXiv:1904.02596.
\item Cook, R. D. (1977). Detection of influential observation in linear regression. Technometrics, 19(1), 15-18.
\item Iglewicz, B., & Hoaglin, D. C. (1993). How to detect and handle outliers (Vol. 16). Asq Press.
\item Leys, C., Klein, O., Dominicy, Y., \& Ley, C. (2018). Detecting multivariate outliers: Use a robust variant of Mahalanobis distance. Journal of Experimental Social Psychology, 74, 150-156.
\item Liu, F. T., Ting, K. M., & Zhou, Z. H. (2008, December). Isolation forest. In 2008 Eighth IEEE International Conference on Data Mining (pp. 413-422). IEEE.
\item Rousseeuw, P. J., \& Van Zomeren, B. C. (1990). Unmasking multivariate outliers and leverage points. Journal of the American Statistical association, 85(411), 633-639.
}
}
