.packageName <- "phangorn"

.onLoad  <- function(libname, pkgname) {
    library.dynam("phangorn", pkgname, libname)
    require(ape) 
}

"ldfactorial" <- function(x){
	x = (x+1)/2
	res = lgamma(2*x)-(lgamma(x)+(x-1)*log(2))
	res
}


"dfactorial" <- function(x){exp(ldfactorial(x))}


#
# Hadamard Conjugation
#

hadamard <- function(x){
	res=1
	while(x>0){
		res=rbind(cbind(res,res),cbind(res,-res))
		x=x-1
	}
	res
}


fhm <- function(v){
	n = length(v)
	n = log2(n)
	res = .C("fhm", v = as.double(v), n = as.integer(n),PACKAGE = "phangorn")$v
	res
}


seq2split = function(s){
	n=length(s)
	res= fhm(log(fhm(s)))/n
	res
}


split2seq = function(q){
	n=length(q)
	res= fhm(exp(fhm(q)))/n
	res
}


distanceHadamard <- function(dm){
    if(class(dm) == "dist"){
        n <- attr(dm,"Size")
        Labels=attr(dm, "Labels")  
        }
        
    if(class(dm) == "matrix"){
        n <- dim(dm)[1]
        dm <- dm[lower.tri(dm)]
        Labels <- colnames(dm)
    } 
    ns <- 2^(n-1)
    if (n > 23) stop("Hadamard conjugation works only efficient for n < 24")
    result <- .Call("dist2spectra",dm, as.integer(n), as.integer(ns)) #, PACKAGE = "phangorn")
    res <- data.frame(distances = result, edges = -fhm(result)/2^(n-2))
    attr(result,"Labels") <- Labels
    res
}


h4st = function(obj, levels=c('a','c','g','t')){
	if(is.matrix(obj)) obj = as.data.frame(t(obj))

	DNA = as.data.frame(obj)
	n = dim(DNA)[1]
	p = dim(DNA)[2]

	if(p>11) stop("4-state Hadamard conjugation works only efficient for n < 12")

	DNAX = matrix(0,n,p)
	DNAY = matrix(0,n,p)

	DNAX[DNA==levels[1]]=0
	DNAX[DNA==levels[2]]=1
	DNAX[DNA==levels[3]]=1
	DNAX[DNA==levels[4]]=0

	DNAY[DNA==levels[1]]=0
	DNAY[DNA==levels[2]]=1
	DNAY[DNA==levels[3]]=0
	DNAY[DNA==levels[4]]=1

	DNAY = DNAY - DNAY[,p]
	DNAX = DNAX - DNAX[,p]

	DNAY = abs(DNAY[,-p])
	DNAX = abs(DNAX[,-p])
	dy = DNAY %*% (2^(0:(p-2))) 
	dx = DNAX %*% (2^(0:(p-2))) 

	INDEX =  dx + 2^(p-1) * dy
	blub = table(INDEX)
	index = as.numeric(rownames(blub)) + 1
	sv = numeric(4^(p-1))
	sv[index] = blub
	qv = matrix(seq2split(sv),2^(p-1),2^(p-1))
	sv = matrix(sv,2^(p-1),2^(p-1))

	q = cbind(transversion = qv[-1,1], transition.1 = diag(qv)[-1], transition.2 = qv[1,-1])
	result = list(q = q, qv = qv, sv=sv, n=sum(sv), names=names(obj))
	result
}


h2st = function (obj, levels = c("r", "y")) 
{
    if (is.matrix(obj)) 
        obj = as.data.frame(t(obj))
    obj = as.data.frame(obj)
    n = dim(obj)[1]
    p = dim(obj)[2]
    if (p > 23) 
        stop("Hadamard conjugation works only efficient for n < 24")
    DNAX = matrix(0, n, p)
    DNAX[obj == levels[1]] = 0
    DNAX[obj == levels[2]] = 1
    DNAX = DNAX - DNAX[, p]
    DNAX = abs(DNAX[, -p])
    dx = DNAX %*% (2^(0:(p - 2)))
    INDEX = dx 
    blub = table(INDEX)
    index = as.numeric(rownames(blub)) + 1
    sv = numeric(2^(p - 1))
    sv[index] = blub
    qv = seq2split(sv)
    result = data.frame(edges = qv, splits=sv)
    attr(result,"Labels") = names(obj)
    result
}




write.nexus.splits = function(obj, file = ""){
    dec2bin <- function(x){
        res=" "
        i = 1
        while(x>0){
            if(x %% 2) res = paste(res, as.character(i))
            x = x %/% 2 
            i=i+1
	    }
        res
    }   
    splits <- lapply(as.numeric(dimnames(obj)[[1]])-1, dec2bin)  
    weight <- obj$edges
    taxa.labels <- attr(obj,"Labels")
    ntaxa = length(taxa.labels)
	nsplits = length(splits)
	if(is.null(weight))weight=numeric(nsplits)+100
	cat("#NEXUS\n\n", file = file)
	cat("[Splits block for Spectronet]\n", file = file, append = TRUE)
	cat("[generated by ]\n", file = file, append = TRUE)
	cat(paste("[",attr(citation("phangorn"),"textVersion"),"]\n\n",sep=""), file = file, append = TRUE)
	cat(paste("BEGIN TAXA;\n\tDIMENSIONS NTAX=",ntaxa,";\n", sep = ""), file = file, append = TRUE)
	cat("\tTAXLABELS",paste( taxa.labels,sep=" "),";\nEND;\n\n", file = file, append = TRUE)
	cat(paste("BEGIN ST_SPLITS;\n\tDIMENSIONS NSPLITS=",nsplits,";\n", sep = ""), file = file, append = TRUE)
	cat("\tFORMAT LABELS WEIGHTS;\n\tMATRIX\n", file = file, append = TRUE)
	for(i in 1:nsplits) cat("\t\t",i,weight[i],paste(splits[[i]]),",\n", file = file, append = TRUE)
	cat("\t;\nEND;\n", file = file, append = TRUE)
}


#
# tree distance functions
#

treedist <- function (tree1, tree2) 
{
	tree1 = unroot(tree1)
    tree2 = unroot(tree2)
    symmetric.difference = NULL
    branch.score.difference = NULL
    path.difference = NULL
    quadratic.path.difference = NULL

    o1 = order(tree1$tip.label)
    o2 = order(tree2$tip.label)
    ll = length(o1)
    p1 = bipartition(tree1)
    p2 = bipartition(tree2)
    p = dim(p1)[1]
    M1 = p1[, o1]
    M2 = p2[, o2]
    if (!is.null(tree1$edge.length) & !is.null(tree2$edge.length)) {
        v1 = tree1$edge.length
        v2 = tree2$edge.length
        dv1 = t(M1 * v1) %*% ((1 - M1) * v1) + t((1 - M1) * v1) %*% 
            (M1 * v1)
        dv2 = t(M2 * v2) %*% ((1 - M2) * v2) + t((1 - M2) * v2) %*% 
            (M2 * v2)
        quadratic.path.difference = sqrt(sum((dv1 - dv2)^2)/2)
    }
    R = M1 %*% t(M2) + (1 - M1) %*% t(1 - M2)
    R = (R%%ll == 0)
    r1 = rowSums(R) > 0
    r2 = colSums(R) > 0
    symmetric.difference = 2 * (p - sum(r1))
    if (!is.null(tree1$edge.length) & !is.null(tree2$edge.length)) {
        v1 = tree1$edge.length
        v2 = tree2$edge.length
        ind1 <- (1:p)[r1]
        ind2 <- unlist(apply(R, 1, which, TRUE))
        s1 = sum((v2[ind2] - v1[ind1])^2)
        zaehler = abs(v2[ind2] - v1[ind1])
        nenner = (v2[ind2] + v1[ind1])/2
        difference = matrix(0, sum(r1), 4)
        difference[, 1] = zaehler
        difference[, 2] = nenner
        difference[, 3] = ind1
        difference[, 4] = ind2
        s2 = sum((v1[(1:p)[!r1]])^2)
        s3 = sum((v2[(1:p)[!r2]])^2)
        branch.score.difference = sqrt(s1 + s2 + s3)
    }
    M1[M1 == 0] = -1
    M2[M2 == 0] = -1
    dt1 = (p - t(M1) %*% M1)/2
    dt2 = (p - t(M2) %*% M2)/2
    path.difference = sqrt(sum((dt1 - dt2)^2)/2)
    result = list(symmetric.difference = symmetric.difference, 
        branch.score.difference = branch.score.difference, path.difference = path.difference, 
        quadratic.path.difference = quadratic.path.difference)
    class(result)="treedist"
    result              
}


print.treedist <-
function(x,...){
    cat("Symmetric difference:", x$symmetric.difference, "\n")
    cat("Branch score difference:", x$branch.score.difference, "\n")
    cat("Path difference:", x$path.difference, "\n")
    cat("Weighted path difference:", x$quadratic.path.difference, "\n")
}



#
# UPGMA, NJ and UNJ
#

"upgma" <- function(D,method="average",...){
	DD=as.dist(D)
	hc = hclust(DD,method=method,...)
	result = as.phylo(hc)
	result = reorder(result, "pruningwise")
	result$edge.length=result$edge.length/2
	result
}


NJ <- function (x) 
{
    x = as.matrix(x)
    labels <- attr(x, "Labels")[[1]]
    edge.length = NULL
    edge = NULL
    d = as.matrix(x)
    if (is.null(labels)) 
        labels = colnames(d)
    l = dim(d)[1]
    m = l - 2
    nam = 1:l
    k = 2 * l - 2
    while (l > 2) {
        r = rowSums(d)/(l - 2)
        i = 0
        j = 0
        tmp <- .C("out", as.double(d), as.double(r), as.integer(l), 
            as.integer(i), as.integer(j), PACKAGE = "phangorn")
        e2 = tmp[[5]]
        e1 = tmp[[4]]
        l1 = d[e1, e2]/2 + (r[e1] - r[e2])/(2)
        l2 = d[e1, e2] - l1
        edge.length = c(l1, l2, edge.length)
        edge = rbind(c(k, nam[e2]), edge)
        edge = rbind(c(k, nam[e1]), edge)
        nam = c(nam[c(-e1, -e2)], k)
        dnew = (d[e1, ] + d[e2, ] - d[e1, e2])/2
        d = cbind(d, dnew)
        d = rbind(d, c(dnew, 0))
        d = d[-c(e1, e2), -c(e1, e2)]
        k = k - 1
        l = l - 1
    }
    edge.length = c(d[2, 1], edge.length)
    attr(edge.length,"names") = NULL
    result = list(edge = rbind(c(nam[2], nam[1]), edge), edge.length = edge.length,
     tip.label = labels, Nnode = m)
    class(result) <- "phylo"
    # test 
    reorder(result, "pruningwise")
}


UNJ = function (x) 
{
    x = as.matrix(x)
    labels <- attr(x, "Labels")[[1]]
    edge.length = NULL
    edge = NULL
    d = as.matrix(x)
    if (is.null(labels)) 
        labels = colnames(d)
    l = dim(d)[1]
    n = l
    nam = as.character(1:l)
    m=l-2
	nam = 1:l
	k = 2*l-2       
    w = rep(1,l)
    while (l > 2) {
        r = rowSums(d)/(l - 2)
        i = 0
        j = 0
        tmp <- .C("out", as.double(d), as.double(r), as.integer(l), as.integer(i), as.integer(j), PACKAGE = "phangorn")
        e2 = tmp[[5]]
        e1 = tmp[[4]]
        l1 = d[e1, e2]/2 + sum((d[e1,-c(e1,e2)] - d[e2,-c(e1,e2)])*w[-c(e1,e2)])/(2*(n-w[e1]-w[e2]))
        l2 = d[e1, e2]/2 + sum((d[e2,-c(e1,e2)] - d[e1,-c(e1,e2)])*w[-c(e1,e2)])/(2*(n-w[e1]-w[e2]))
        edge.length = c(l1, l2, edge.length)
        edge = rbind(c(k, nam[e2]), edge)
        edge = rbind(c(k, nam[e1]), edge)
        nam = c(nam[c(-e1, -e2)], k)
      
        dnew = (w[e1]*d[e1, ] + w[e2]*d[e2, ] - w[e1]*l1 - w[e2]*l2)/(w[e1] + w[e2])
        d = cbind(d, dnew)
        d = rbind(d, c(dnew, 0))
        d = d[-c(e1, e2), -c(e1, e2)]
        w = c(w, w[e1] + w[e2])
        w = w[-c(e1, e2)]
        k = k - 1
        l = l - 1
    }
    edge.length=c(d[2,1],edge.length)
    result = list(edge = rbind(c(nam[2], nam[1]), edge), 
    edge.length=edge.length, tip.label = labels, Nnode=m)
    class(result) <- "phylo"
    reorder(result, "pruningwise")  
}


"dist.hamming" <- function (x, ratio = TRUE) 
{
    if(class(x)!='phyDat') stop("x has to be element of class phyDat")
    l = length(x)
    weight <- attr(x, 'weight')
    d = numeric((l * (l - 1))/2)
    k = 1
    for (i in 1:(l - 1)) {
        for (j in (i + 1):l) {
            d[k] = sum( weight * (rowSums(x[[i]] * x[[j]])==0))
            k = k + 1
        }
    }
    if (ratio) 
        d = d/sum(weight)
    attr(d, "Size") <- l
    if (is.list(x)) 
        attr(d, "Labels") <- names(x)
    else attr(d, "Labels") <- colnames(x)
    attr(d, "Diag") <- FALSE
    attr(d, "Upper") <- FALSE
    attr(d, "call") <- match.call()
    attr(d, "method") <- "hamming"
    class(d) <- "dist"
    return(d)
}



"dist.logDet" <- function(x){
    if(class(x)!='phyDat') stop("x has to be element of class phyDat")
    weight <- attr(x, 'weight')
    r <- attr(x,"nc")
    l = length(x)
    d = numeric((l * (l - 1))/2)
    k = 1
    for (i in 1:(l - 1)) {
        for (j in (i + 1):l) {
            tmp = crossprod(weight * x[[i]],x[[j]])
			class(tmp) = 'matrix'
			if(is.nan( log(det(tmp)) ) ){
			d[k] = 10
			}	
            else d[k] = (-log(det(tmp)) + sum(log(rowSums(tmp) * colSums(tmp)))/2)/r
            k = k + 1
        }
    }
    attr(d, "Size") <- l
    if (is.list(x)) 
        attr(d, "Labels") <- names(x)
    else attr(d, "Labels") <- colnames(x)
    attr(d, "Diag") <- FALSE
    attr(d, "Upper") <- FALSE
    attr(d, "call") <- match.call()
    attr(d, "method") <- "logDet"
    class(d) <- "dist"
    return(d)
}



#
# Data structures for ML and MP
# 

fast.table <- function (data)                                                            
{                                                                                        
	if(is.list(data))data = as.data.frame(data, stringsAsFactors=FALSE)                    
	da = do.call("paste", c(data, sep = "\r"))                                             
	ind = !duplicated(da)                                                                  
	levels = da[ind]                                                                       
	cat <- factor(da,levels = levels)                                                      
	nl <- length(levels(cat))                                                        
	bin <- (as.integer(cat) - 1)                                                           
	pd <- nl                                                                               
	bin <- bin[!is.na(bin)]                                                                
	if (length(bin)) bin <- bin + 1                                                        
	y <- tabulate(bin, pd)                                                                 
	result=list(index = bin, weights = y, data = data[ind,])	                                                                              
	result                                                                                 
}                                                                                        
                                                                   

phyDat.DNA = function (data, return.index = FALSE) 
{
    if (class(data) == "DNAbin") 
        data = as.character(data)
    if (is.matrix(data)) 
        data = as.data.frame(t(data))
    ac = c("a", "c", "g", "t", "u", "m", "r", "w", "s", "y", 
        "k", "v", "h", "d", "b", "n", "?", "-")
    AC = matrix(c(c(1, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 
        0, 1, 1, 1), c(0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 
        0, 1, 1, 1, 1), c(0, 0, 1, 0, 0, 0, 1, 0, 1, 0, 1, 1, 
        0, 1, 1, 1, 1, 1), c(0, 0, 0, 1, 1, 0, 0, 1, 0, 1, 1, 
        0, 1, 1, 1, 1, 1, 1)), 18, 4, 
        dimnames = list(NULL, c("a", "c", "g", "t")))
    data = as.data.frame(data, stringsAsFactors = FALSE)
    nam = names(data)
    ddd = fast.table(data)
    data = ddd$data
    index = ddd$index
    q = length(data)
    p = length(data[[1]])
    tmp <- vector("list", q)
    for (i in 1:q) tmp[[i]] = factor(data[[i]], levels = ac)
    data <- tmp
    for (i in 1:q) class(data[[i]]) = "integer"
    class(data) <- "data.frame"
    row.names(data) = as.character(1:p)
    data = na.omit(data)
    rn = as.numeric(rownames(data))
    ind = which(!duplicated(c(rn, as.character(unique(index)))))
    ind = ind - length(rn)
    ind = ind[ind > 0]
    for (i in 1:length(ind)) index[which(index == ind[i])] = NA
    indextmp = diff(sort(unique(index)))
    l1 = which(indextmp > 1)
    d1 = indextmp[l1]
    if (length(l1) > 0) {
        for (i in 1:length(l1)) {
            index[index > l1[i] & !is.na(index)] = index[index > 
                l1[i] & !is.na(index)] - (d1[i] - 1)
        }
    }
    weight = ddd$weight[rn]
    p = dim(data)[1]
    dat = list()
    for (i in 1:q) dat[[i]] = AC[data[[i]],]
    names(dat) = nam
    attr(dat, "weight") = weight
    attr(dat, "nr") = p
    attr(dat, "nc") = 4
    if (return.index) 
        attr(dat, "index") = index
    attr(dat, "levels") = c("a", "c", "g", "t")
    class(dat) = "phyDat"
    dat
}


as.phyDat <- function (x, ...) UseMethod("as.phyDat")

as.phyDat.DNAbin <- function(data,...) phyDat.DNA(data,...)


as.data.frame.phyDat <- function(x, ...){
        lev = attr(x,"levels")
        fn = function(x, levels, y) paste(levels[x*y], collapse="")               
        nr = attr(x, "nr")
        nc = attr(x, "nc")
        y = 1:nc
        X = matrix(nrow=nr, ncol=length(x))
        for(i in 1:length(x)) X[,i]= apply(x[[i]], 1, fn, lev, y)
        if(is.null(attr(x,"index"))) index=rep(1:nr, attr(x,"weight"))
        else index = attr(x,"index")
        result = X[index,]
        colnames(result) = names(x)
        result = as.data.frame(result, stringsAsFactors = FALSE)
        result
}


as.character.phyDat <- function (x, ...) 
{
    lev = attr(x, "levels")
    fn = function(x, levels, y) paste(levels[x * y], collapse = "")
    nr = attr(x, "nr")
    nc = attr(x, "nc")
    y = 1:nc
    X = matrix(nrow = length(x), ncol = nr)
    for (i in 1:length(x)) X[i, ] = apply(x[[i]], 1, fn, lev, y)
    if (is.null(attr(x, "index"))) index = rep(1:nr, attr(x, "weight"))
    else index = attr(x, "index")
    result = X[ ,index]
    rownames(result) = names(x)
    result
}


phyDat = function(data, levels, return.index=FALSE){
	# new ape format
	if(class(data)=="DNAbin") data = as.character(data)
	# old ape format 
	if(is.matrix(data)) data = as.data.frame(t(data))                                                                   
	data = as.data.frame(data, stringsAsFactors=FALSE)
    nam = names(data) 	                                     
	ddd = fast.table(data)                                                                 
	data = ddd$data                                                                        
	weight = ddd$weight  
	index = ddd$index                                                                
  	q = length(data)                                                                       
    p = length(data[[1]])
    tmp <- vector("list", q)
    for (i in 1:q) tmp[[i]] = factor(data[[i]], levels = levels)
    data <- tmp
    for (i in 1:q) class(data[[i]]) = "integer"
    class(data) <- "data.frame"    
                                                                       
    row.names(data) = as.character(1:p)
    data = na.omit(data)
    rn = as.numeric(rownames(data))
    ind = which(!duplicated(c(rn,as.character(unique(index)))))
	ind = ind-length(rn)
	ind = ind[ind>0]
    for(i in 1:length(ind)) index[which(index==ind[i])] = NA
    indextmp = diff(sort(unique(index)))
	l1 = which(indextmp>1)
	d1 = indextmp[l1]
	if(length(l1)>0){
		for(i in 1:length(l1)){
			index[index>l1[i] & !is.na(index)] = index[index>l1[i] & !is.na(index)] - (d1[i]-1)
		}
	}	   
	weight = ddd$weight[rn]
                                                               
    p = dim(data)[1]                                                                     
    dat = list()                                                                         
    l = length(levels)                                                                   
    AACC = diag(l)                                                                       
    for (i in 1:q) dat[[i]] = matrix(unlist(AACC[,data[[i]]], TRUE, FALSE), ncol = l, 
    	byrow = TRUE)                                                                            
    names(dat) = nam                     
    attr(dat,"weight") = weight  
    attr(dat,"nr") = p
    attr(dat,"nc") = length(levels)
    if(return.index) attr(dat,"index") = index
    attr(dat, "levels") = levels                                                             
    class(dat) = "phyDat"
    dat                                                                               
}


print.phyDat = function (x, ...) 
{
    cat(length(x), "sequences with",sum(attr(x,"weight")), "character and",attr(x,"nr"),"different site patterns.\n")
    cat("The states are",attr(x,"levels"), "\n")
}


baseFreq <- function(dat){
    if (class(dat) != "phyDat") 
        stop("data must be of class phyDat")
    levels <- attr(dat,"levels")
    weight <- attr(dat,"weight")	
    n <- length(dat)	
    res <- numeric(length(levels)) 	
    for(i in 1:n)res <- res+colSums(dat[[i]]*weight)	
    res <- res/sum(res)
    names(res) <- levels	
    res	
}


phylo <- function(edge, tip, edge.length=NULL){
    res <- list(edge=edge, tip.label=tip, edge.length=edge.length)
	class(res)="phylo"
	res
	}
	
	
phyloNode <- function(root, pvector, cvector, evector, tip, tips, Nnode){
	res <- list(root=root, pvector=pvector, cvector=cvector, evector=evector, tip=tip, tips=tips, Nnode=Nnode)
	class(res)="phyloNode"
	res
	}
	

as.phyloNode.phylo <- function(object, ...){
    parents <- object$edge[,1]
    child <- object$edge[,2]
    if (is.null(attr(object, "order")) || attr(object, "order") == "cladewise") root <- parents[1]
    else root <- parents[length(parents)]
    pvector <- numeric(max(parents))
    pvector[child] <- parents
    tips  <- !logical(max(parents))
    tips[parents] <-  FALSE
    cvector <- vector("list",max(parents))   
    for(i in 1:length(parents))  cvector[[parents[i]]] <- c(cvector[[parents[i]]], child[i]) 
    evector <- NULL 
    if(!is.null(object$edge.length)){
        evector <- numeric(max(parents)) 
        evector[child] <- object$edge.length
    } 
    list(root=root, pvector=pvector, cvector=cvector, evector=evector, tip=object$tip.label, tips=tips, Nnode=object$Nnode)
}
 
as.phylo <- function (x, ...) UseMethod("as.phylo")
as.phyloNode <- function (x, ...) UseMethod("as.phyloNode")

as.phylo.phyloNode <- function(object, order="pruningwise", ...){
	if(order=="pruningwise")result <- phyloPruning(object)
	if(order=="cladewise")result <- phyloClade(object)
	result
}

phyloClade <- function(phyloNode){
    edge <- NULL
    root <- phyloNode$root
    tips <- phyloNode$tips
    cvector <- phyloNode$cvector
    pvector <- phyloNode$pvector
    mylist <- cvector[[root]]
    edge.length <- NULL
    while(length(mylist)){
        kid = mylist[1]
    	edge <- rbind(edge, cbind(pvector[kid],kid))
    	if(!is.null(phyloNode$evector)) edge.length <- c(edge.length, phyloNode$evector[kid])
        if(!tips[kid]) mylist <- c(cvector[[kid]], mylist[-1])     
        else mylist <- mylist[-1]  
        } 
    tree <- list(edge=edge, edge.length=edge.length, tip.label=phyloNode$tip, Nnode=phyloNode$Nnode)
    class(tree) <- "phylo" 
    attr(tree, "order") <- "cladewise"
    tree
}


phyloPruning <- function(phyloNode){
    edge <- NULL
    root <- phyloNode$root
    tips <- phyloNode$tips
    cvector <- phyloNode$cvector
    parent.list <- root
    edge.length <- NULL
    while(length(parent.list)){
        parent <- parent.list[1]
        kids <- cvector[[parent]]
    	edge <- rbind(cbind(parent,kids),edge)
    	if(!is.null(phyloNode$evector)) edge.length <- c(phyloNode$evector[kids], edge.length)
    	parent.list = c(parent.list[-1], kids[which(!tips[kids])] )
    }
    tree <- list(edge=edge, edge.length=edge.length, tip.label=phyloNode$tip, Nnode=phyloNode$Nnode)
    class(tree) <- "phylo" 
    attr(tree, "order") <- "pruningwise"
    tree 
}


getCols <- function (data, cols) 
{
    attrib = attributes(data)
    attr(data, "class") <- "list"
    data = data[cols]
    if (is.character(cols)) 
        attrib$names = cols
    else attrib$names = attrib$names[cols]
    attributes(data) = attrib
    attr(data, "class") <- "phyDat" 
    data
}

getRows <- function (data, rows) 
{    
    for (i in 1:length(data)) data[[i]] = as.matrix(data[[i]])[rows, ]
    attr(data, "weight") = attr(data, "weight")[rows]
    attr(data, "nr") = length(rows)
    data
}


#
# Maximum Parsimony 
# 

 
sankoff.quartet <- function (dat, cost, p, l, weight) 
{
    tmp <- .Call("sankoffQuartet", sdat = dat, sn = p, scost = cost, 
        sk = l) #, PACKAGE = "phangorn")
    erg <- .Call("rowMin", tmp, as.integer(p), as.integer(l), PACKAGE = "phangorn")
    sum(weight * erg)
}

 
sankoffNNI <- function (tree, n, datp, datf, p, l, p0, cost, weight) 
{
    edge = matrix(tree$edge, ncol = 2)
    parent = edge[, 1]
    child = tree$edge[, 2]
    k = min(parent) - 1
    nTips = min(parent) - 1
    ind = which(child > nTips)[n]
    p1 = parent[ind]
    p2 = child[ind]
    ind1 = which(parent == p1)
    ind1 = ind1[ind1 != ind]
    ind1 = c(which(child == p1), ind1)
    ind2 = which(parent == p2)
    e1 = child[ind1[1]]
    if (p1 > k + 1) 
        e1 = parent[ind1[1]]
    e2 = child[ind1[2]]
    e3 = child[ind2[1]]
    e4 = child[ind2[2]]
    datn = vector("list", 4)
    attr(datn, "dim") = c(1, 4)
    if (p1 == k + 1) 
        datn[[1]] = datf[[e1]]
    if (p1 > k + 1) 
        datn[[1]] = datp[[p1]]
    datn[[2]] = datf[[e2]]
    datn[[3]] = datf[[e3]]
    datn[[4]] = datf[[e4]]
    datt = datn[, c(1, 3, 2, 4)]
    attr(datt, "dim") = c(1, 4)
    new1 <- sankoff.quartet(datt, cost, p, l, weight)
    datt = datn[, c(1, 4, 3, 2)]
    attr(datt, "dim") = c(1, 4)
    new2 <- sankoff.quartet(datt, cost, p, l, weight)
    res = c(p0, new1, new2)
    wm = which.min(res)
    edgeID = NULL
    swap = FALSE
    if (wm > 1) {
        swap = TRUE
        edgeID = c(ind1, ind2, ind)
    }
    list(res = res, edgeID = edgeID, swap = swap, wm = wm)
}





parsimony <- function(tree, data, method='sankoff',...){
    if(is.rooted(tree))tree <- unroot(tree)
    if(is.null(attr(tree,"order")) || attr(tree,"order")=="cladewise")tree <- reorder(tree, "pruningwise")  
    if(method=='sankoff') result <- sankoff(tree,data,...)
    if(method=='fitch') result <- fitch(tree,data)
    result 
}


fitch <- function (tree, data) 
{
    if (class(data) != "phyDat") 
        stop("data must be of class phyDat")
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "pruningwise")
    levels <- attr(data, "levels")
    l = length(levels)
    weight = attr(data, "weight")
    p = attr(data, "nr")
    q = length(data)
    data <- prepareDataFitch(data)
    d = attributes(data)
    data <- as.integer(data)
    attributes(data) <-d

    node <- tree$edge[, 1]
    edge <- tree$edge[, 2]
    m = length(edge) + 1
    dat = integer(m * p)
    attr(dat, "dim") <- c(m, p)
    dat[1:q, ] = data[tree$tip.label, ]
    pars <- integer(p)
    result <- .C("fitch3", dat, as.integer(p), as.integer(m), 
        as.integer(pars), as.integer(node), as.integer(edge), 
        as.integer(length(edge)))
    sum(weight * result[[4]])
}




prepareDataFitch <- function(data){
    lev <- attr(data,"levels")
    l <- length(lev)
    nr <- attr(data,"nr")  
    X <- matrix(ncol=nr, nrow=length(data))
    for(i in 1:length(data)) X[i,] = data[[i]] %*% 2^c(0:(l-1))
    attrData <- attributes(data)
    nam <- attrData$names
    attrData$names <- NULL
    X <- as.integer(X)
    attributes(X) <- attrData 
    attr(X, "dim") <- c(length(data), nr)
    dimnames(X) <- list(nam,NULL)
    X
}


prepareDataSankoff <- function(data){
    tf = function(dat) {
        dat[dat == 0] = 1e+06
        dat[dat == 1] <- 0
        dat
    }   
    attrData <- attributes(data)
    data <- lapply(data, tf)
    attributes(data) <- attrData 
    data
}


sankoff <- function (tree, data, cost = NULL) 
{
    if (class(data) != "phyDat") 
        stop("data must be of class phyDat")
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "p")
    data <- prepareDataSankoff(data)

    levels <- attr(data, "levels")
    l = length(levels)  

    if (is.null(cost)) {
        cost <- matrix(1, l, l)
        cost <- cost - diag(l)
    }   
    for (i in 1:length(data)) storage.mode(data[[i]]) = "double"
    fit.sankoff(tree, data, cost, FALSE)
}


fit.sankoff <- function (tree, data, cost, returnData = FALSE) 
{
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "p")
    node <- tree$edge[, 1]
    edge <- tree$edge[, 2]
    weight = attr(data, "weight")
    p = attr(data, "nr")
    q = length(tree$tip.label)
    l = attr(data, "nc")
    m = length(edge) + 1
    dat = vector(mode = "list", length = m)
    dat[1:q] = data[tree$tip.label]
    nr = as.integer(dim(dat[[1]])[1])
    nc = as.integer(dim(dat[[1]])[2])
    node = as.integer(node - 1)
    edge = as.integer(edge - 1)
    nTips = as.integer(length(tree$tip))
    mNodes = as.integer(max(node) + 1)
    tips = as.integer((1:length(tree$tip))-1)
    res <- .Call("sankoff3", dat, as.numeric(cost), as.integer(nr),as.integer(nc),
         node, edge, mNodes, tips, PACKAGE="phangorn")  
    root <- node[length(node)] + 1
    erg <- .Call("rowMin", res[[root]], as.integer(p), as.integer(l), PACKAGE = "phangorn")
    pscore <- sum(weight * erg)
    result = pscore
    if (returnData) 
        result <- list(pscore = pscore, dat = res)
    result
}



pnodes <- function (tree, dat, cost, external = TRUE) 
{
    dl = dim(dat[[1]])[2]
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "p")
    l = length(dat)
    parent <- tree$edge[, 1]
    child <- tree$edge[, 2]
    datp = vector("list", l)
    pl = length(parent) + 1
    p = dim(dat[[1]])[1]
    pj = parent[pl - 1]
    start = pl - 1
    tmp = dat[[1]] * 0
    datp[[pj]] = tmp
    nTips = min(parent) - 1
    for (j in (pl - 1):1) {
        blub = TRUE
        isParent = (child[j] > nTips)
        if (!external & !isParent) 
            blub = FALSE
        if (blub) {
            res <- .Call("sankoff2", sdat = datp[[parent[j]]], 
                sn = p, scost = cost, sk = dl, PACKAGE = "phangorn")
            if (pj != parent[j]) {
                pj = parent[j]
                start = j
            }
            i = start
            while (i > 0 && pj == parent[i]) {
                if (i != j) 
                  res <- res + .Call("sankoff2", sdat = dat[[child[i]]], 
                    sn = p, scost = cost, sk = dl, PACKAGE = "phangorn")
                i = i - 1
            }
            datp[[child[j]]] = res
        }
    }
    datp
}


sankoff.nni <- function (tree, data, cost, ...) 
{
    nnimove2 <- function(tree, id, first) {
        child = tree$edge[, 2]
        if (first) {
            tree$edge[id[2], 2] = child[id[3]]
            tree$edge[id[3], 2] = child[id[2]]
        }
        if (!first) {
            tree$edge[id[2], 2] = child[id[4]]
            tree$edge[id[4], 2] = child[id[2]]
        }
        tree
    }
    if (class(data) != "phyDat") 
        stop("data must be of class phyDat")
    levels <- attr(data, "levels")
    l = length(levels)
    weight = attr(data, "weight")
    p = attr(data, "nr")
    kl = TRUE
    i = 1
    tmp = fit.sankoff(tree, data, cost, TRUE)
    p0 = tmp[[1]]
    datf = tmp[[2]]
    datp = pnodes(tree, datf, cost, FALSE)
    swap = 0
    result = NULL
    indM = NULL
    ll = NULL
    ind = NULL
    id <- NULL
    wm <- NULL
    tmp = numeric(length(tree$edge[, 1]))
    while (kl) {
        res = sankoffNNI(tree, n = i, datp = datp, datf = datf, 
            p, l, p0, cost, weight)
        result = rbind(result, res[[1]])
        if (res$swap) {
            swap = TRUE
            tmp2 = tmp
            tmp2[res$edgeID] = 1
            indM = rbind(indM, tmp2)
            ll = c(ll, res$res[res$wm])
            id = rbind(id, res$edgeID)
            wm = c(wm, res$wm)
        }
        if (i == (tree$Nnode - 1)) 
            kl = FALSE
        i = i + 1
    }
    if (swap) {
        l = length(ll)
        INDEX = matrix(0, l, length(tree$edge[, 1]))
        for (i in 1:length(ll)) INDEX[i, id[i, ]] = 1
        ind = which.min(ll)
        rll = rank(ll, ties.method = "random")
        tmp = which(tcrossprod(INDEX)[, ind] == 0)
        while (length(tmp) > 0) {
            st = tmp[which.min(rll[tmp])]
            INDEX[ind[1], ] = INDEX[ind[1], ] + INDEX[st, ]
            ind = c(ind, st)
            tmp = which(tcrossprod(INDEX)[, ind[1]] == 0)
        }
        swap = 0
        for (i in ind) {
            tree2 <- nnimove2(tree, id[i, ], wm[i] == 2)
            tree3 <- as.phylo.phyloNode(as.phyloNode.phylo(tree2))
            p1 = fit.sankoff(tree3, data, cost)
            if (p1 < p0) {
                swap = swap + 1
                tree = tree2
                p0 = p1
            }
        }
        cat(swap, "\n")
        tree <- as.phylo.phyloNode(as.phyloNode.phylo(tree))
    }
    list(tree = tree, pscore = p0, swap = swap)
}


optim.parsimony <- function(tree,data,cost=NULL,...) {
    if(is.rooted(tree))tree <- unroot(tree)
    if(is.null(attr(tree,"order")) || attr(tree,"order")=="cladewise")tree <- reorder(tree, "pruningwise")
    
    dat <- prepareDataSankoff(data)
    l <- attr(dat, "nc")
    if (is.null(cost)) {
        cost <- matrix(1, l, l)
        cost <- cost - diag(l)
    }
    tree$edge.length=NULL
            swap = 0
            iter = TRUE
            pscore <- fit.sankoff(tree,dat,cost)
            while (iter) {
                res <- sankoff.nni(tree,dat,cost,...)
                tree <- res$tree
                cat("optimize topology: ", pscore , "-->", res$pscore, 
                  "\n")
                pscore = res$pscore
                swap = swap + res$swap
                if (res$swap == 0) iter = FALSE
            }
            cat("Final p-score",pscore,"after ",swap, "nni operations \n") 
            list(tree=tree,pscore=pscore)          
}



#
# Maximum likelihood estimation
#

incomplete.gamma <- function(x,p){
#	oflo = 1e+30
	acu = 1e-8
	g = lgamma(p)
	pn = numeric(6)
	gin = 0.0
	fact = exp(p*log(x)-x-g)

	if(x<1 | x<p){
		gin = 1.0
		term = 1.0
		rn = p

		while(term > acu){
			rn = rn + 1.0
			term = term * x / rn
			gin = gin + term
		}
		gin = gin * fact / p
		return(gin)
	}

	else{
		dif=1
		rn=1
		a = 1-p
		b = a  + x + 1
		term = 0
		pn[1] = 1 
		pn[2] = x
		pn[3] = x+1
		pn[4] = x*b
		gin = pn[3] * pn[4]
		while(dif > acu*rn){
			a = a+1
			b = b+2
			term = term+1
			an = a *term
			for(i in 1:2)pn[i+4] = b*pn[i+2] - an*pn[i]
			if(abs(pn[6])<1e-6)pn[6]=1e-6
			rn = pn[5]/pn[6]
			dif = abs(gin-rn)
			gin = rn
			for(i in 1:4) pn[i] = pn[i+2]
		}
		gin = 1 - fact * gin
		return(gin)
	} 
	gin
}


discrete.gamma = function(alpha, k){
	if(k==1)return(1)
	quants = qgamma((1:(k-1))/k,shape = alpha,rate = alpha)
	lower = c(0,unlist(lapply(quants * alpha,incomplete.gamma,alpha+1))) 
	upper = c(unlist(lapply(quants * alpha,incomplete.gamma,alpha+1)),1)
	result  <- (upper-lower)*k
	result
}


optimQ = function(tree, data, Q=c(1,1,1,1,1,1), trace=0,...){
	l = length(Q)
	Q = Q[-l]
	Q = sqrt(Q)
	fn = function(Q,tree,data,...){
	  pml2(tree, data, Q=c(Q^2,1),...)
	}
	res = optim(par=Q, fn=fn, gr=NULL, method = "L-BFGS-B", lower=0, upper=Inf, control=list(fnscale=-1, maxit=25, trace=trace),tree=tree, data=data,...)
	res[[1]] = c(res[[1]]^2,1)	
	res
	}	
		
    
optimGamma = function(tree, data, shape=1, k=4,...){
	fn = function(shape, tree, data, k,...)pml2(tree, data, shape=shape, k=k,...)
	res = optimize(f=fn, interval = c(0,100), lower = 0, upper = 100, maximum = TRUE,
		 tol = .01, tree=tree, data=data, k=k,...)
	res
	}
    
 
optimInv = function(tree, data, inv=0.01, INV=NULL, ll.0=NULL,...){
	fn = function(inv, tree, data,...)pml2(tree, data, inv=inv, INV=INV, ll.0=NULL,...)
	res = optimize(f=fn, interval = c(0,1), lower = 0, upper = 1, maximum = TRUE,
		 tol = .0001, tree=tree, data=data,...)
	res
	}
  

optimRate <- function(tree, data, rate=1, ...){
    fn <- function(rate, tree, data, ...) pml2(tree, data, rate=rate, ...)
    res <- optimize(f = fn, interval = c(0, 100), lower = 0, upper = 100, 
        maximum = TRUE, tol = 0.01, tree = tree, data = data, ...)
    res
}
		
	
optimBf = function(tree, data, bf=c(.25,.25,.25,.25), ll.0=NULL, trace=0,...){
	l=length(bf)
	nenner = 1/bf[l]
	lbf = log(bf * nenner)
	lbf = lbf[-l]
	fn = function(lbf, tree, data,...){
		bf = exp(c(lbf,0))
		bf = bf/sum(bf)
		pml2(tree, data, bf=bf, ll.0=NULL,...)
		}
	res = optim(par=lbf, fn=fn, gr=NULL, method="Nelder-Mead", control=list(fnscale=-1, maxit=500, trace=trace),tree=tree, data=data,...)
	bf = exp(c(res[[1]],0))
	bf = bf/sum(bf)
	result = list(bf=bf, loglik = res[[2]])
	result
	}


optimW = function(fit,...){
	w = fit$w
	g = fit$g
	siteLik = fit$siteLik
	k = length(w)
	l = dim(siteLik[[1]])[1]
	x=matrix(0,l,k)
	for(i in 1:k)x[,i] = rowSums(siteLik[[i]])
	weight = fit$weight
	nenner = 1/w[k]
	eta = log(w * nenner)
	eta = eta[-k]
	
	fn = function(eta,x,g,weight){
		eta = c(eta,0)
		p = exp(eta)/sum(exp(eta))
		res = x%*%p
		res = sum(weight*log(res))  * (1 + abs(sum(p*g) - 1))
		res
	}  
	res = optim(eta, fn = fn, method = "Nelder-Mead", control=list(fnscale=-1, reltol = 1e-12),gr=NULL, x=x,g=g, weight=weight)
	p = exp(c(res$par,0))
	p = p/sum(p)
	result = list(par = p, value = res$value)
	result	
}


predict.pml <- function(object, newdata,...) sum(object$site * newdata)


logLik.pml <- function(object,...){
	res <- object$logLik
	attr(res,"df") <- object$parameter$df
	class(res) <- "logLik"
	res
}


print.pml = function(x,...){
	param = x$parameter
	cat("\nloglikelihood:",x$logLik,"\n\n")
	if(param$inv > 0)cat("Proportion of invariant sites:",param$inv,"\n")
	if(param$k >1){
		cat("Discrete gamma model\n")
		cat("Number of rate categories:",param$k,"\n")		
		cat("Shape parameter:",param$shape,"\n")
	}
}
	
	
summary.pml = function(object,...){
	res = list()
	res$parameter = object$parameter
	res$weight = object$weight
	class(res) = "summary.pml"
	res
}	
	
	
vcov.pml <- function(object,...){
	FI = score(object,...)[[2]]
	l = dim(FI)[1]
	res = try(solve(FI))
	if(class(res) == "try-error"){
	    cat("Covariance is ill-conditioned !! \n")
		res = solve(FI + diag(l)* 1e-8)
		}
	res
}
		

print.summary.pml = function(x,...){
	param <- x$parameter
	cat("\nloglikelihood:", param$logLik, "\n")
    w <- x$weight
    w <- w[w>0]	
    ll0 = sum(w*log(w/sum(w)))
    cat("\nunconstrained loglikelihood:", ll0, "\n\n")
	if(param$inv > 0)cat("Proportion of invariant sites:",param$inv,"\n")
	if(param$k >1){
		cat("Discrete gamma model\n")
		cat("Number of rate categories:",param$k,"\n")		
		cat("Shape parameter:",param$shape,"\n")
		}
	cat("\nRate matrix:\n")	
	QM = matrix(0,length(param$levels), length(param$levels), dimnames = list(param$levels,param$levels))	
	QM[lower.tri(QM)] = param$Q	
	QM = QM+t(QM)
	print(QM)
	cat("\nBase frequencies:  \n")
	bf = param$bf
	names(bf) = param$levels
	print(bf)	
	cat("\n")
}	

# generateSP <- sitePattern
allSitePattern <- function(s,levels=c("a","c","g","t"), names=NULL){
	l=length(levels)
	X=matrix(0, l^s,s)
	for(i in 1:s)
	X[, i] = rep(rep(c(1:l), each=l^(i-1)),l^(s-i))
	for(i in 1:l)X[X==i] = levels[i]
	if(is.null(names))colnames(X) = paste("t",1:s, sep="")
	else colnames(X)=names
	phyDat(t(X), levels)
}	

	
write.phylip <- function(data, weight, file=""){
        n = sum(weight)
        m = dim(data)[2]
        cat(m,n,"\n",file = file)
        for(i in 1:m)
        cat(colnames(data)[i],"   ",toupper(rep(data[,i],weight)),"\n", sep="", file=file, append=TRUE)
}


getd2P <- function(el, eig=edQt(), g=1.0){
	n <- length(eig$values)	
	res <- .Call("getd2PM",eig,as.integer(n),as.double(el),as.double(g), PACKAGE = "phangorn")
    attr(res,"dim") <- c(length(g),length(el))
	res
}	
	
getdP <- function(el, eig=edQt(), g=1.0){
	n <- length(eig$values)	
	res <- .Call("getdPM",eig,as.integer(n),as.double(el),as.double(g), PACKAGE = "phangorn")
    attr(res,"dim") <- c(length(g),length(el))
	res
}


# version without transformation 
getdP2 <- function(el, eig=edQt(), g=1.0){
	n <- length(eig$values)	
	res <- .Call("getdPM2",eig,as.integer(n),as.double(el),as.double(g), PACKAGE = "phangorn")
    attr(res,"dim") <- c(length(g),length(el))
	res
}

getP <- function(el, eig=edQt(), g=1.0){
	n <- length(eig$values)	
	res <- .Call("getPM",eig,as.integer(n),as.double(el),as.double(g), PACKAGE = "phangorn")
	attr(res,"dim") <- c(length(g),length(el))
	res
}




lli <- function (data, tree, bf = c(0.25, 0.25, 0.25, 0.25), ...) 
{
    p = attr(data,"nr")
    dat = data[tree$tip.label]
    m = length(dat)
    l = length(bf)
    res <- matrix(1, p, l)
    for (i in 1:m) res = res * dat[[i]]
    res
}


edQt <- function (Q = c(1, 1, 1, 1, 1, 1), bf = c(0.25, 0.25, 0.25, 0.25)) 
{
    l = length(bf)
    res = matrix(0, l, l)
    res[lower.tri(res)] = Q
    res = res + t(res)
    res = res * bf
    res2 = res * rep(bf, each = l)    
    diag(res) = -colSums(res)
    res = res/sum(res2)
    e = eigen(res, FALSE)
    e$inv = solve(e$vec)
    e
}


edQ <- function(Q=c(1,1,1,1,1,1), bf=c(0.25,.25,.25,.25)){
	l=length(bf)
    res = matrix(0, l, l)
	res[lower.tri(res)] = Q
	res = res+t(res)
	res = res * rep(bf,each=l)
	diag(res) = -rowSums(res)
	res2 = res * rep(bf,l)
	diag(res2)=0 
	res = res/sum(res2)
	e = eigen(res, FALSE)
	e$inv = solve(e$vec)
	e
}



pml3 <- function (object,...) 
{
	tree = object$tree
    para = object$parameter
    Q = para$Q
    bf = para$bf
    eig = para$eig
    w = para$w
    g = para$g
    data = object$data
    ll0 <- object$logLik
    ll.0 <- object$ll.0
    weight = attr(data, "weight")
    lll = ll.0
    m = 1
    p = length(g)
    q = length(tree$edge[, 1]) + 1
    resll = vector("list", p)
    while (m <= p) {
        res = ll(data, tree, bf = bf, g = g[m], Q = Q, eig = eig, 
            assign.dat = FALSE, ...)
        resll[[m]] <- res
        m = m + 1
    }
    for (i in 1:p) lll = lll + resll[[i]] * w[i]
    siteLik = log(lll)
    ll0 = sum(weight * siteLik)
    ll0
}


ll <- function (dat1, tree, bf = c(0.25, 0.25, 0.25, 0.25), g = 1, 
    Q = c(1, 1, 1, 1, 1, 1), eig = NULL, assign.dat = FALSE, ...) 
{
	if(is.null(attr(tree,"order")) || attr(tree,"order")=="cladewise")tree <- reorder(tree, "pruningwise")
    q = length(tree$tip.label) 
    node <- tree$edge[, 1]
    edge <- tree$edge[, 2]
    m = length(edge) + 1
    dat = vector(mode = "list", length = m)
    dat[1:q] = dat1[tree$tip.label]
    if (is.null(eig)) eig = edQt(bf = bf, Q = Q)
    el <- tree$edge.length
    P <- getP(el, eig, g)  
    
    nr = as.integer(dim(dat[[1]])[1])
    nc = as.integer(dim(dat[[1]])[2])
    node = as.integer(node-min(node))
    edge = as.integer(edge-1)
    nTips = as.integer(length(tree$tip))
    mNodes = as.integer(max(node)+1)
        
    res <- .Call("LogLik", dat[1:q], P, nr, nc, node, edge, nTips, mNodes, PACKAGE = "phangorn")

    result = res[[1]] %*% bf  
    if (assign.dat){
    	dat[(q+1):m] <- res
        attr(dat, "names") = c(tree$tip.label, as.character((q + 1):m))
        assign("asdf", dat, env = parent.frame(n = 1))
        }
    result
}



fs <- function (old.el, eig, parent.dat, child.dat, weight, g=g, 
    w=w, bf=bf, ll.0=ll.0) 
{
    if (old.el < 1e-10) 
        old.el <- 1e-10
    lg = length(parent.dat)
    dad <- vector("list", lg)
    P <- getP(old.el, eig, g)
    for (i in 1:lg) dad[[i]] <- parent.dat[[i]]/(child.dat[[i]] %*% P[[i]])

    f = ll.0
    for (i in 1:lg) f = f + parent.dat[[i]] %*% (w[i] * bf)
    l0 = sum(weight * log(f))

    .Call("FS", eig, as.integer(length(bf)), as.double(old.el), 
            as.double(w), as.double(g), child.dat, dad, as.integer(length(w)), 
            as.integer(length(weight)), as.double(bf), as.double(weight), 
            as.double(ll.0), as.double(l0), as.double(f), PACKAGE="phangorn" )
}


fn.quartet <- function(old.el, eig, bf, dat,  g=1, w=1, weight, ll.0) {
	l= length(dat[,1]) 
    ll = ll.0
    res = vector("list", 2*l)
    tmp1 = NULL
    tmp2 = NULL
    attr(res,"dim") = c(l,2)
    
    for(j in 1:l){
    		P = getP(old.el, eig, g[j])
    		tmp1 = (dat[[j,1]] %*% P[[1]]) *(dat[[j,2]] %*% P[[2]])
    		tmp2 = (dat[[j,3]] %*% P[[3]]) * (dat[[j,4]] %*% P[[4]])
    		res[[j,1]] = tmp1 * (tmp2 %*% P[[5]])
    		res[[j,2]] = tmp2
      		ll = ll +  res[[j,1]] %*% (w[j]*bf)
        } 
 	l0 = sum(weight * log(ll))
 	list(ll=l0,res=res)
}


nnimove <- function(tree, id, el, first){
	child = tree$edge[, 2]
	if(first){
        tree$edge[id[2], 2] = child[id[3]]
        tree$edge[id[3], 2] = child[id[2]]
        tree$edge.length[id] = el
	}
	if(!first){
        tree$edge[id[2], 2] = child[id[4]]
        tree$edge[id[4], 2] = child[id[2]]
        tree$edge.length[id] = el        
	}
	tree
}


pml.nni <- function (fit, ...) 
{
    tree = fit$tree
    kl = TRUE
    i = 1
    .dat <- NULL
    datp = rnodes(fit)
    swap = 0
    result = NULL
    indM = NULL
    ll = NULL
    el = NULL
    ind = NULL
    id <- NULL
    wm <- NULL
    tmp = numeric(length(tree$edge.length))
    while (kl) {
        res = phyloNNI(fit, n = i, datp, datf = .dat)
        result = rbind(result, res[[1]])
        if (res$swap) {
            swap=TRUE
            tmp2 = tmp
            tmp2[res$edgeID] = 1
            indM = rbind(indM, tmp2)
            ll = c(ll, res$res[res$wm])
            el = rbind(el, res$para)
            id = rbind(id, res$edgeID)
            wm = c(wm, res$wm)
        }
        if (i == (tree$Nnode - 1)) 
            kl = FALSE
        i = i + 1
    }
    if(swap) {
        l = length(ll)
        INDEX = matrix(0, l, length(tree$edge.length))
        for (i in 1:length(ll)) INDEX[i, id[i, ]] = 1
        ind = which.max(ll)
        rll = rank(ll, ties.method = "random")
        tmp = which(tcrossprod(INDEX)[, ind] == 0)
        while (length(tmp) > 0) {
            st = tmp[which.max(rll[tmp])]
            INDEX[ind[1], ] = INDEX[ind[1], ] + INDEX[st, ]
            ind = c(ind, st)
            tmp = which(tcrossprod(INDEX)[, ind[1]] == 0)
        }
	    l0=fit$log
	    fit2=fit	
	    swap=0
        for (i in ind){
		tree2 <- nnimove(tree, id[i, ], el[i, ], wm[i] == 2)
		tree3 <- as.phylo.phyloNode(as.phyloNode.phylo(tree2)) 
        fit2$tree = tree3
		l1 = pml3(fit2)
		if(l1 > l0){
			swap = swap+1
			tree=tree2
			l0=l1
			}
		}
        cat(swap," \n")  
        tree <- as.phylo.phyloNode(as.phyloNode.phylo(tree))  
        fit$tree <- tree
        fit <- optimEdge(fit, control = list(eps = 1e-08, maxit = 5))
    }
    fit$swap = swap
    fit
}


bipartition <- function(tree) 
{
    bp <- .Call("bipartition", tree$edge, length(tree$tip), tree$Nnode, PACKAGE = "ape")
    nTips = length(tree$tip)	
    l = length(bp)
    m = length(bp[[1]])
    k = length(tree$edge[,1])
    result = matrix(0, l, m)
    res = matrix(0, k, m)
    for (i in 1:l) result[i, bp[[i]]] = 1
    result[result[, 1] == 0, ] = 1 - result[result[, 1] == 0, ]
    result = result[-1, ]
    for(i in 1:nTips)res[which(tree$edge[,2]==i),i]=1	
    res[tree$edge[,2]>nTips,] = result  
    colnames(res) = tree$tip.label
    res
}



rnodes <- function (fit, external = TRUE) 
{
    dat <- NULL
    logLik <- pml5(fit)
    tree = fit$tree
    if (is.null(attr(tree, "order")) || attr(tree, "order") == "cladewise") 
        tree <- reorder(tree, "p")
    eig = fit$parameter$eig
    l = dim(dat)[1]
    parent <- tree$edge[, 1]
    child <- tree$edge[, 2]
    nTips = min(parent) - 1
    pl = length(parent) + 1
    datp = vector("list", pl * l)
    attr(datp, "dim") = c(l, pl)
    dat2 = vector("list", pl * l)
    attr(dat2, "dim") = c(l, pl)
    el = tree$edge.length
    gs = fit$g
    datp[, (nTips + 1)] = dat[, (nTips + 1)]
    for (j in (pl - 1):1) {
        blub=TRUE
        isParent = (child[j] > nTips)
        if(!external & !isParent) blub=FALSE 
        if(blub){
            elx = el[j]
            P = getP(elx, eig, gs)
        
            for (i in 1:l) {
                tmp2 = (datp[[i, parent[j]]]/(dat[[i, child[j]]] %*% P[[i]]))
                dat2[[i, child[j]]] = tmp2
                if (isParent) 
                    datp[[i, child[j]]] = (tmp2 %*% P[[i]]) * dat[[i, child[j]]]
            }
        } 
    }
    assign(".dat", dat, env = parent.frame(n = 1))
    dat2
}


score <- function (fit) 
{
    tree = fit$tree
    eig = fit$parameter$eig
    .dat <- NULL
    datp = rnodes(fit,TRUE)
    m = dim(.dat)[1]
    g = fit$g
    bf = fit$parameter$bf
    w = fit$w
    parent <- tree$edge[, 1]
    child <- tree$edge[, 2]
    nTips = min(parent) - 1
    l = length(child)
    sc = numeric(l)
    weight = fit$weight
    f <- fit$ll.0
    for (j in 1:m) f = f + (w[j] * .dat[[j, (nTips + 1)]]) %*% bf
    
    dl = matrix(0,length(weight), l)

    ff0 = numeric(length(weight))
    for (i in 1:l) {
        elx = tree$edge.length[i]
        ff = ff0
        dP = getdP(elx, eig, g)
        for (j in 1:m) {
                    ff = ff +  (datp[[j, child[i] ]] * (.dat[[j, child[i] ]] %*% dP[[j]])) %*% (w[j] * bf)
        }
        dl[,i] = ff/f
        sc[i] = sum(weight * dl[,i])
    }       
    F = crossprod(dl*weight,dl) 
    names(sc) = child
    dimnames(F) = list(child, child) 
    result = list(sc = sc, F = F)
    result
}



optim.quartet <- function (old.el, eig, bf, dat, g = 1, w = 1, weight, ll.0 = weight * 
    0, control = list(eps = 1e-04, maxit = 5, trace = 0), llcomp=-Inf) 
{
    tmp <- fn.quartet(old.el = old.el, eig = eig, bf = bf, dat = dat, 
        g = g, w = w, weight = weight, ll.0 = ll.0)
    old.ll = tmp$ll
    eps = 1
    iter = 0
    while (eps > control$eps && iter < control$maxit) {
        el1 <- fs(old.el[1], eig, tmp$res[, 1], dat[, 1], weight, 
            g = g, w = w, bf = bf, ll.0 = ll.0)
        el2 <- fs(old.el[2], eig, el1[[2]], dat[, 2], weight, 
            g = g, w = w, bf = bf, ll.0 = ll.0)
        el5 <- fs(old.el[5], eig, el2[[2]], tmp$res[, 2], weight, 
            g = g, w = w, bf = bf, ll.0 = ll.0)
        el3 <- fs(old.el[3], eig, el5[[3]], dat[, 3], weight, 
            g = g, w = w, bf = bf, ll.0 = ll.0)
        el4 <- fs(old.el[4], eig, el3[[2]], dat[, 4], weight, 
            g = g, w = w, bf = bf, ll.0 = ll.0)
        old.el[1] = el1[[1]]
        old.el[2] = el2[[1]]
        old.el[3] = el3[[1]]
        old.el[4] = el4[[1]]
        old.el[5] = el5[[1]]
        iter = iter + 1
        tmp <- fn.quartet(old.el = old.el, eig = eig, bf = bf, 
            dat = dat, g = g, w = w, weight = weight, ll.0 = ll.0)
        ll = tmp[[1]]
        eps = ll - old.ll
        if(ll<llcomp)return(list(old.el, ll))  
        old.ll = ll
    }
    list(old.el, ll)
}


phyloNNI <- function (fit, n, datp, datf) 
{
    tree = fit$tree
    para = fit$parameter
    bf = para$bf
    eig = para$eig
    k = para$k
    w = para$w
    g = para$g
    edge = matrix(tree$edge, ncol = 2)
    parent = edge[, 1]
    child = tree$edge[, 2]
    k = min(parent) - 1
    ll.0 <- fit$ll.0
    nTips = min(parent) - 1
    ind = which(child > nTips)[n]
    p1 = parent[ind]
    p2 = child[ind]
    ind1 = which(parent == p1)
    ind1 = ind1[ind1 != ind]
    ind1 = c(which(child == p1), ind1)
    ind2 = which(parent == p2)
    e1 = child[ind1[1]]
    e2 = child[ind1[2]]
    e3 = child[ind2[1]]
    e4 = child[ind2[2]]
    el0 = tree$edge.length[c(ind1[1], ind1[2], ind2[1], ind2[2], 
        ind)]
    l = length(datf[, 1])
    weight = fit$weight
    datn = vector("list", 4 * l)
    attr(datn, "dim") = c(l, 4)
    if (p1 <= k + 1) 
        datn[, 1] = datf[, e1]
    if (p1 > k + 1) 
        datn[, 1] = datp[, e1]
    datn[, 2] = datf[, e2]
    datn[, 3] = datf[, e3]
    datn[, 4] = datf[, e4]
    datt = datn[, c(1, 3, 2, 4)]
    attr(datt, "dim") = c(l, 4)
    new1 <- optim.quartet(el0[c(1, 3, 2, 4, 5)], eig, bf, datt, 
        g, w, weight, ll.0, llcomp= fit$log)
    datt = datn[, c(1, 4, 3, 2)]
    attr(datt, "dim") = c(l, 4)
    new2 <- optim.quartet(el0[c(1, 4, 3, 2, 5)], eig, bf, datt, 
        g, w, weight, ll.0, llcomp= fit$log)
    res = c(fit$log, new1[[2]], new2[[2]])
    wm = which.max(res)
    edgeID = NULL
    swap = FALSE
    para = NULL
    if (wm > 1) {
        swap = TRUE
        if (wm == 2) 
            para = new1[[1]]
        if (wm == 3) 
            para = new2[[1]]
        edgeID = c(ind1, ind2, ind)
    }
    list(res = res, el = el0, edgeID = edgeID, swap = swap, wm = wm, 
        para = para)
}



plot.pml<-function(x,...)plot.phylo(x$tree,...)



update.pml <- function (object, ...) 
{
    extras <- match.call(expand.dots = FALSE)$...
    pmla <- names(as.list(args(pml)))
    names(extras) <- pmla[pmatch(names(extras), pmla[-length(pmla)])]
    existing <- match(pmla, names(extras))
    if (is.na(existing[1])) 
        tree <- object$tree
    else tree <- eval(extras[[existing[1]]], parent.frame())
    if (is.na(existing[2])) 
        data <- object$data
    else data <- eval(extras[[existing[2]]], parent.frame())
    if (is.na(existing[3])) 
        bf <- object$parameter$bf
    else bf <- eval(extras[[existing[3]]], parent.frame())
    if (is.na(existing[4])) 
        Q <- object$parameter$Q
    else Q <- eval(extras[[existing[4]]], parent.frame())
    inv <- ifelse(is.na(existing[5]), object$parameter$inv, eval(extras[[existing[5]]], 
        parent.frame()))
    k <- ifelse(is.na(existing[6]), object$parameter$k, eval(extras[[existing[6]]], 
        parent.frame()))
    shape <- ifelse(is.na(existing[7]), object$parameter$shape, 
        eval(extras[[existing[7]]], parent.frame()))
    rate <- ifelse(is.na(existing[8]), object$rate, 
        eval(extras[[existing[8]]], parent.frame()))
    levels <- attr(data, "levels")
    weight <- attr(data, "weight")
    m <- 1
    eig <- edQt(bf = bf, Q = Q)
    w <- rep(1/k, k)
    if (inv > 0) 
        w <- (1 - inv) * w
    g <- discrete.gamma(shape, k)
    if (inv > 0) 
        g <- g/(1 - inv)
    g <- rate * g
    INV <- lli(data, tree, bf)
    ll.0 <- INV %*% (bf * inv)
    resll <- vector("list", k)
    while (m <= k) {
        resll[[m]] = ll(data, tree, bf, g[m], Q, eig, assign.dat = FALSE)
        m = m + 1
    }
    lll = ll.0
    for (i in 1:k) lll = lll + resll[[i]] * w[i]
    siteLik = log(lll)
    ll0 = sum(weight * siteLik)   
    df = length(tree$edge.length) + k - 1 + (inv > 0) + length(unique(bf)) - 
        1 + length(unique(Q)) - 1
    parameter = list(logLik = ll0, inv = inv, k = k, shape = shape, 
        g = g, w = w, eig = eig, Q = Q, bf = bf, levels = levels, 
        df = df)
    result = list(logLik = ll0, siteLik = siteLik, weight = weight, 
        g = g, w = w, rate=rate, parameter = parameter, data = data, INV = INV, 
        ll.0 = ll.0, tree = tree, call = call)
    class(result) = "pml"
    
    result
}



phangornParseFormula <- function(model){

    parseSide <- function(model) {
        model.vars <- list()
        while (length(model) == 3 && model[[1]] == as.name("+")) {
            model.vars <- c(model.vars, model[[3]])
            model <- model[[2]]
        }
        unlist(rev(c(model.vars, model)))

    } 

    if (!inherits(model, "formula")) 
        stop("model must be a formula object")
    l <- length(model)
    varsLHS <- NULL
       
    if(l==3){        
    modelLHS <- model[[2]]
    modelRHS <- model[[3]]
    varsRHS <- parseSide(modelRHS)
    varsRHS <- unlist(lapply(varsRHS,as.character))
    varsLHS <- parseSide(modelLHS)
    varsLHS <- unlist(lapply(varsLHS,as.character))
    }
    if(l==2){
       modelRHS <- model[[2]]
       varsRHS <- parseSide(modelRHS)
       varsRHS <- unlist(lapply(varsRHS,as.character))
    }
    list(left=varsLHS, right=varsRHS)
}




# added lv als likelihood vektor 
pml <- function (tree, data, bf = NULL, Q = NULL, inv = 0, k = 1, shape = 1, 
    rate = 1, ...) 
{
    call <- match.call()
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "pruningwise")
    if (class(data)[1] != "phyDat") 
        stop("data must be of class phyDat")
    levels <- attr(data, "levels")
    weight <- attr(data, "weight")
    if (is.null(bf)) 
        bf <- rep(1/length(levels), length(levels))
    if (is.null(Q)) 
        Q <- rep(1, length(levels) * (length(levels) - 1)/2)
    m <- 1
    eig <- edQt(bf = bf, Q = Q)
    w <- rep(1/k, k)
    if (inv > 0) 
        w <- (1 - inv) * w
    g <- discrete.gamma(shape, k)
    if (inv > 0) 
        g <- g/(1 - inv)
    g <- rate * g
    INV <- lli(data, tree, bf)
    ll.0 <- INV %*% (bf * inv)
    resll <- vector("list", k)
    while (m <= k) {
        resll[[m]] = ll(data, tree, bf = bf, g = g[m], Q = Q, 
            eig = eig, assign.dat = FALSE, ...)
        m = m + 1
    }
    lll <- numeric(length(ll.0))
    
    lll <- lll + ll.0
    for (i in 1:k) lll = lll + resll[[i]] * w[i]
    siteLik <- lll
    siteLik <- log(siteLik)
    loglik = sum(weight * siteLik)
    df = length(tree$edge.length) + k - 1 + (inv > 0) + length(unique(bf)) - 
        1 + length(unique(Q)) - 1
    parameter = list(logLik = loglik, inv = inv, k = k, shape = shape, 
        g = g, w = w, eig = eig, Q = Q, bf = bf, levels = levels, 
        df = df)
    result = list(logLik = loglik, siteLik = siteLik, weight = weight, 
        g = g, w = w, rate = rate, parameter = parameter, data = data, 
        INV = INV, ll.0 = ll.0, tree = tree, lv=lll, call=call)
    class(result) = "pml"
    result
}


optW <- function (ll, weight, omega,...) 
{
    k = length(omega)
    nenner = 1/omega[1]
    eta = log(omega * nenner)
    eta = eta[-1]
	
    fn = function(eta, ll, weight) {
        eta = c(0,eta)
        p = exp(eta)/sum(exp(eta))
        res = sum(weight * log(ll %*% p)) 
        res
    }
    if(k==2)res = optimize(f =fn , interval =c(-3,3) , lower = -3, upper = 3, maximum = TRUE, tol = .Machine$double.eps^0.25, ll = ll, weight = weight) 
    else res = optim(eta, fn = fn, method = "Nelder-Mead", control = list(fnscale = -1, 
        reltol = 1e-12), gr = NULL, ll = ll, weight = weight)
    p = exp(c(0,res[[1]]))
    p = p/sum(p)
    result = list(par = p, value = res[[2]])
    result
}


pml.control <- function (epsilon = 1e-06, maxit = 10, trace = FALSE) 
{
    if (!is.numeric(epsilon) || epsilon <= 0) 
        stop("value of 'epsilon' must be > 0")
    if (!is.numeric(maxit) || maxit <= 0) 
        stop("maximum number of iterations must be > 0")
    list(epsilon = epsilon, maxit = maxit, trace = trace)
}


optim.pml <- function (object, optNni = FALSE, optBf = FALSE, optQ = FALSE, 
    optInv = FALSE, optGamma = FALSE, optEdge = TRUE, optRate = FALSE, 
    control = pml.control(maxit = 10, eps = 0.001, trace=TRUE)) 
{
    tree = object$tree
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "pruningwise")
    if (optNni) {
        if (!is.binary.tree(tree)) 
            tree = multi2di(tree)
    }
    if (is.rooted(tree)) {
        tree = unroot(tree)
        warning("I rooted the tree (unrooted trees are not yet supported)", 
            call. = FALSE)
    }
    if (optEdge & optRate) {
        warning("you can't optimise edges and rates at the same time, only edges are optimised", 
            call. = FALSE)
        optRate = FALSE
    }
    trace <- control$trace 
    para = object$parameter
    Q = para$Q
    bf = para$bf
    eig = para$eig
    inv = para$inv
    k = para$k
    shape = para$shape
    w = para$w
    g = para$g
    dat = object$data
    ll0 <- object$logLik
    INV <- object$INV
    ll.0 <- object$ll.0
    rate <- object$rate
    ll = ll0
    ll1 = ll0
    opti = TRUE
    if (optEdge) {
        object <- optimEdge(object, control = pml.control(eps = 0.001, maxit = 5,trace))
        ll <- object$logLik
        tree <- object$tree
    }
    rounds = 1
    df = 0
    while (opti) {
        if (optBf) {
            res = optimBf(tree, dat, bf = bf, inv = inv, Q = Q, 
                w = w, g = g, INV = INV, rate = rate)
            bf = res[[1]]
            eig = edQt(Q = Q, bf = bf)
            ll.0 = INV %*% (bf * inv)
            cat("optimize base frequencies: ", ll, "-->", res[[2]], 
                "\n")
            ll = res[[2]]
        }
        if (optQ) {
            res = optimQ(tree, dat, Q = Q, bf = bf, w = w, g = g, 
                inv = inv, INV = INV, ll.0 = ll.0, rate = rate)
            Q = res[[1]]
            eig = edQt(Q = Q, bf = bf)
            cat("optimize Q: ", ll, "-->", res[[2]], "\n")
            ll = res[[2]]
        }
        if (optInv) {
            res = optimInv(tree, dat, inv = inv, INV = INV, Q = Q, 
                bf = bf, eig = eig, k = k, shape = shape, rate = rate)
            inv = res[[1]]
            w = rep(1/k, k)
            g = discrete.gamma(shape, k)
            if (inv > 0) {
                w = (1 - inv) * w
                g = g/(1 - inv)
            }
            g <- g * rate
            ll.0 = INV %*% (bf * inv)
            cat("optimize invariant sites: ", ll, "-->", res[[2]], 
                "\n")
            ll = res[[2]]
        }
        if (optGamma) {
            res = optimGamma(tree, dat, shape = shape, k = k, 
                inv = inv, INV = INV, Q = Q, bf = bf, eig = eig, 
                ll.0 = ll.0, rate = rate)
            shape = res[[1]]
            w = rep(1/k, k)
            g = discrete.gamma(shape, k)
            if (inv > 0) {
                w = (1 - inv) * w
                g = g/(1 - inv)
            }
            g <- g * rate
            cat("optimize shape parameter: ", ll, "-->", res[[2]], 
                "\n")
            ll = res[[2]]
        }
        if (optRate) {
            res = optimRate(tree, dat, inv = inv, INV = INV, 
                Q = Q, bf = bf, eig = eig, k = k, shape = shape, 
                rate = rate)
            rate = res[[1]]
            w = rep(1/k, k)
            g = discrete.gamma(shape, k)
            if (inv > 0) {
                w = (1 - inv) * w
                g = g/(1 - inv)
            }
            g <- g * rate
            cat("optimize rate: ", ll, "-->", res[[2]], "\n")
            ll = res[[2]]
        }
        if (optEdge) {
            object <- pml(tree = tree, dat = dat, Q = Q, bf = bf, 
                inv = inv, shape = shape, k = k, rate = rate)
            object <- optimEdge(object, control = list(eps = 0.001, maxit = 5))
            ll <- object$logLik
            tree <- object$tree
        }
        if (optNni) {
            swap = 0
            iter = 1
            while (iter < 4) {
                tree <- object$tree
                object <- pml.nni(object)
                tree <- object$tree
                cat("optimize topology: ", ll, "-->", object$logLik, 
                  "\n")
                ll = object$logLik
                swap = swap + object$swap
                iter = iter + 1
                if (object$swap == 0) {
                  iter = 4
                  optNni = FALSE
                }
            }
            cat(swap, "\n")
            if (swap > 0) 
                rounds = 1
            if (swap == 0) 
                optNni = FALSE
        }
        rounds = rounds + 1
        if (rounds > control$maxit) 
            opti = FALSE
        if (abs(ll1 - ll) < control$eps) 
            opti = FALSE
        ll1 = ll
    }
    df = (optEdge) * length(tree$edge.length) + optGamma * (k - 
        1) + optInv * (inv > 0) + optBf * (length(unique(bf)) - 
        1) + optQ * (length(unique(Q)) - 1)
    object <- pml(tree = tree, dat = dat, Q = Q, bf = bf, inv = inv, 
        shape = shape, k = k, rate = rate)
    object$parameter$df = df
    object
}


pml2 <- function (tree, data, bf = rep(1/length(levels), length(levels)), 
    shape = 1, k = 1, Q = rep(1, length(levels) * (length(levels) - 
        1)/2), levels = attr(data, "levels"), inv = 0, rate = 1, 
    g = NULL, w = NULL, eig = NULL, INV = NULL, ll.0 = NULL, ...) 
{
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "pruningwise")
    if (class(data)[1] != "phyDat") 
        stop("data must be of class phyDat")
    weight = attr(data, "weight")
    l = length(bf)
    lll = matrix(0, length(weight), l)
    m = 1
    if (is.null(eig)) 
        eig = edQt(bf = bf, Q = Q)
    if (is.null(w)) {
        w = rep(1/k, k)
        if (inv > 0) 
            w <- (1 - inv) * w
    }
    if (is.null(g)) {
        g = discrete.gamma(shape, k)
        if (inv > 0) 
            g <- g/(1 - inv)
    }
    g <- g * rate
    if (is.null(INV)) 
        INV = lli(data, tree, bf)
    if (is.null(ll.0)) 
        ll.0 = INV %*% (inv * bf)
    lll <- ll.0
    p = length(g)
    resll = vector("list", p)
    while (m <= p) {
        resll[[m]] = ll(data, tree, bf = bf, g = g[m], Q = Q, eig = eig, ...)
        m = m + 1
    }
    for (i in 1:p) lll = lll + resll[[i]] * w[i]
    siteLik = log(lll)
    result = sum(weight * siteLik)
    result
}

optimEdge <- function (fit, control = list(eps = 1e-08, maxit = 50, trace = 0), 
    ...) 
{
    if (class(fit)[1] != "pml") 
        stop("data must be of class pml")
    tree = fit$tree
    if (is.null(attr(tree, "order")) || attr(tree, "order") == 
        "cladewise") 
        tree <- reorder(tree, "pruningwise")
    fit$tree <- tree
    el <- tree$edge.length
    tree$edge.length[el < 0] <- 1e-08
    dat <- NULL    
    rate <- fit$rate 
    old.ll <- pml5(fit)
    eig <- fit$parameter$eig
    w <- fit$w
    g <- fit$g

    bf <- fit$parameter$bf
    weight <- attr(fit$data, "weight")
    ll.0 <- fit$ll.0
    eps = 1
    iter = 0
    child = tree$edge[, 2]
    parent = tree$edge[, 1]
    nTips = min(parent) - 1
    n = length(tree$edge.length)
    while (eps > control$eps && iter < control$maxit) {
        for (j in n:1) {
            child.dat = dat[, child[j]]
            parent.dat = dat[, parent[j]]
            old.el = tree$edge.length[j]
            newEL <- fs(old.el, eig, parent.dat, child.dat, weight, 
                g = g, w = w, bf = bf, ll.0 = ll.0)
            el[j] = newEL[[1]]
            dat[, parent[j]] = newEL[[2]]
            if (child[j] > nTips) {
                dat[, child[j]] = newEL[[3]]
            }
        }
        tree$edge.length = el
        iter = iter + 1
        fit$tree = tree
        dat <- NULL
        newll <- pml5(fit)
        eps = newll - old.ll
        old.ll = newll
    }
    cat(fit$logLik, " -> ", newll, "\n")
    fit$logLik = newll
    fit
}

pml5<-function (object, ...) 
{
    tree = object$tree
    para = object$parameter
    Q = para$Q
    bf = para$bf
    eig = para$eig
    w = para$w
    g = para$g
    data = object$data
    ll0 <- object$logLik
    ll.0 <- object$ll.0
    weight = attr(data, "weight")
    lll = ll.0
    m = 1
    p = length(g)
    q = length(tree$edge[, 1]) + 1
    resll = vector("list", p)
    dat = vector("list", q * p)
    attr(dat, "dim") = c(p, q)
    asdf <- NULL
    while (m <= p) {
        resll[[m]] = ll(data, tree, bf = bf, g = g[m], Q = Q, 
            eig = eig, assign.dat = TRUE, ...)
        dat[m, ] <- asdf
        m = m + 1
    }
    attr(dat, "dimnames") = list(NULL, attr(asdf, "names"))
    for (i in 1:p) lll = lll + resll[[i]] * w[i]
    siteLik <- lll
    siteLik <- log(siteLik)
    ll0 = sum(weight * siteLik)
    assign("dat", dat, env = parent.frame(n = 1))
    ll0
}


