#' plot.fitRMU plots the results of a fit RMU.
#' @title Plot the synthesis of RMU fit.
#' @author Marc Girondot
#' @return Return Nothing
#' @param x A result file generated by fitRMU
#' @param ... Parameters used by plot
#' @param what Can be proportions, numbers or total
#' @param aggregate Can be model of both
#' @param order Give the order of series in plot, from bottom to top. Can be used to not show series.
#' @param control.legend Parameters send to legend
#' @param show.legend If FALSE, does not show legend
#' @description The function plot.fitRMU plots the results of fitRMU().
#' @family Fill gaps in RMU
#' @examples
#' \dontrun{
#' library("phenology")
#' RMU.names.AtlanticW <- data.frame(mean=c("Yalimapo.French.Guiana", 
#'                                          "Galibi.Suriname", 
#'                                          "Irakumpapy.French.Guiana"), 
#'                                  se=c("se_Yalimapo.French.Guiana", 
#'                                       "se_Galibi.Suriname", 
#'                                       "se_Irakumpapy.French.Guiana"))
#' data.AtlanticW <- data.frame(Year=c(1990:2000), 
#'       Yalimapo.French.Guiana=c(2076, 2765, 2890, 2678, NA, 
#'                                6542, 5678, 1243, NA, 1566, 1566),
#'       se_Yalimapo.French.Guiana=c(123.2, 27.7, 62.5, 126, NA, 
#'                                  230, 129, 167, NA, 145, 20),
#'       Galibi.Suriname=c(276, 275, 290, NA, 267, 
#'                        542, 678, NA, 243, 156, 123),
#'       se_Galibi.Suriname=c(22.3, 34.2, 23.2, NA, 23.2, 
#'                            4.3, 2.3, NA, 10.3, 10.1, 8.9),
#'       Irakumpapy.French.Guiana=c(1076, 1765, 1390, 1678, NA, 
#'                                3542, 2678, 243, NA, 566, 566),
#'       se_Irakumpapy.French.Guiana=c(23.2, 29.7, 22.5, 226, NA, 
#'                                  130, 29, 67, NA, 15, 20))
#'                            
#' cst <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'                colname.year="Year", model.trend="Constant", 
#'                model.SD="Zero")
#' expo <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'                colname.year="Year", model.trend="Exponential", 
#'                model.SD="Zero")
#' YS <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'              colname.year="Year", model.trend="Year-specific", 
#'              model.SD="Zero")
#' YS1 <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'              colname.year="Year", model.trend="Year-specific", 
#'              model.SD="Zero", model.rookeries="First-order")
#' YS1_cst <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'              colname.year="Year", model.trend="Year-specific", 
#'              model.SD="Constant", model.rookeries="First-order", 
#'              parameters=YS1$par)
#' YS2 <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'              colname.year="Year", model.trend="Year-specific",
#'              model.SD="Zero", model.rookeries="Second-order", 
#'              parameters=YS1$par)
#' YS2_cst <- fitRMU(RMU.data=data.AtlanticW, RMU.names=RMU.names.AtlanticW, 
#'              colname.year="Year", model.trend="Year-specific",
#'              model.SD="Constant", model.rookeries="Second-order", 
#'              parameters=YS1_cst$par)
#'                
#' compare_AIC(Constant=cst, Exponential=expo, 
#' YearSpecific=YS)
#' 
#' compare_AIC(YearSpecific_ProportionsFirstOrder_Zero=YS1,
#' YearSpecific_ProportionsFirstOrder_Constant=YS1_cst)
#' 
#' compare_AIC(YearSpecific_ProportionsConstant=YS,
#'            YearSpecific_ProportionsFirstOrder=YS1,
#'            YearSpecific_ProportionsSecondOrder=YS2)
#'            
#' compare_AIC(YearSpecific_ProportionsFirstOrder=YS1_cst,
#'            YearSpecific_ProportionsSecondOrder=YS2_cst)
#' 
#' barplot_errbar(YS1_cst$proportions[1, ], y.plus = YS1_cst$proportions.CI.0.95[1, ], 
#' y.minus = YS1_cst$proportions.CI.0.05[1, ], las=1, ylim=c(0, 0.7), 
#' main="Proportion of the different rookeries in the region")
#' 
#' plot(cst, main="Use of different beaches along the time", what="total")
#' plot(expo, main="Use of different beaches along the time", what="total")
#' plot(YS2_cst, main="Use of different beaches along the time", what="total")
#' 
#' plot(YS1, main="Use of different beaches along the time")
#' plot(YS1_cst, main="Use of different beaches along the time")
#' plot(YS1_cst, main="Use of different beaches along the time", what="numbers")
#' 
#' parpre <- par(mar=c(4, 4, 2, 5)+0.4)
#' par(xpd=TRUE)
#' plot(YS, main="Use of different beaches along the time", 
#' control.legend=list(x=2000, y=0.4, legend=c("Yalimapo", "Galibi", "Irakumpapy")))
#' par(mar=parpre)
#' 
#' # Example to modify order of series
#' plot(cst, order=c("Galibi.Suriname", "Irakumpapy.French.Guiana"))
#' plot(cst, order=c("Galibi.Suriname", "Irakumpapy.French.Guiana", "Yalimapo.French.Guiana"))
#' }
#' @method plot fitRMU
#' @export

plot.fitRMU <- function (x, ..., what = "proportions", aggregate = "both", order = NULL, 
                         control.legend = list(), show.legend = TRUE) 
{
  what <- tolower(what)
  aggregate <- tolower(aggregate)
  what <- match.arg(what, choices = c("proportions", "numbers", 
                                      "total"))
  aggregate <- match.arg(aggregate, choices = c("both", "model"))
  matobs <- as.matrix(x$RMU.data[, as.character(x$RMU.names[, 
                                                            "mean"])])
  if (any(colnames(x$RMU.names) == "se")) {
    matobsse <- as.matrix(x$RMU.data[, as.character(x$RMU.names[, 
                                                                "se"])])
  }    else {
    matobsse <- matobs
    matobsse[] <- ifelse(is.na(matobsse), NA, matobsse)
  }
  model <- as.matrix(x$numbers)
  modelse <- as.matrix(x$numbers - x$numbers.CI.0.025)/2
  if (what == "total") {
    year <- as.numeric(rownames(x[["proportions"]]))
    if (aggregate == "model") {
      if (x$model.trend == "year-specific") {
        y <- x$par[paste0("T_", year)]
        erry <- 2 * x$SE[paste0("T_", year)]
        y.minus <- y - erry
        y.plus <- y + erry
      }
      if (x$model.trend == "constant") {
        y <- rep(x$par["T_"], length(year))
        erry <- rep(2 * x$SE["T_"], length(year))
        y.minus <- y - erry
        y.plus <- y + erry
      }
      if (x$model.trend == "exponential") {
        y <- x$par["T_"] * exp(x$par["r"] * (1:length(year)))
        y.minus <- (x$par["T_"] - 2 * x$SE["T_"]) * exp((x$par["r"] - 
                                                           2 * x$SE["r"]) * (1:length(year)))
        y.plus <- (x$par["T_"] + 2 * x$SE["T_"]) * exp((x$par["r"] + 
                                                          2 * x$SE["r"]) * (1:length(year)))
      }
    }        else {
      finalobs <- ifelse(is.na(matobs), model, matobs)
      finalobsse <- ifelse(is.na(matobs), modelse, matobsse)
      y <- rowSums(finalobs)
      y.se <- rowSums(finalobsse)
      y.plus <- y + 1.96 * y.se
      y.minus <- y - 1.96 * y.se
    }
    maxy <- max(c(y.plus, y), na.rm = TRUE)
    pp <- modifyList(list(ylim = c(0, maxy), xlab = "Year", 
                          ylab = "Nest number", las = 1, bty = "n", main = "Total number of nests in all the rookeries in the region"), 
                     list(...))
    pp <- modifyList(pp, list(x = year, y = y, y.plus = y.plus, 
                              y.minus = y.minus))
    do.call(plot_errbar, pp)
  }    else {
    if (is.null(order)) 
      order <- colnames(x[[what]])
    xplot <- x[[what]][, order]
    year <- as.numeric(rownames(xplot))
    pp <- modifyList(list(ylim = c(0, max(rowSums(xplot))), 
                          xlab = "Year", ylab = what, las = 1, bty = "n"), 
                     list(...))
    pp <- modifyList(pp, list(x = year, y = xplot[, 1], type = "n"))
    do.call(plot, pp)
    p <- cbind(rep(0, dim(xplot)[1]), rep(0, dim(xplot)[1]), 
               xplot)
    xx <- c(year, rev(year))
    for (i in (1:length(order)) + 1) {
      y <- c(apply(p[, 1:i], 1, sum), rev(apply(p[, 1:(i + 
                                                         1)], 1, sum)))
      polygon(xx, y, col = rainbow(dim(xplot)[2])[i - 1], 
              border = NA)
    }
    if (show.legend) {
      plegend <- modifyList(list(x = "bottomright", legend = rev(order), 
                                 col = rev(rainbow(length(order))), pch = 15), 
                            control.legend)
      do.call(legend, args = plegend)
    }
  }
}
