#' @name sprinkle_sanitize
#' @title Sanitize Characters for LaTeX Outputs
#' 
#' @description Certain characters in LaTeX code need to be escaped to 
#'   prevent errors during processing.  For example, \code{\%} is the 
#'   comment character in LaTeX, and needs to be escaped in 
#'   order to render correctly.  
#'   
#' @param x An object of class \code{dust}
#' @param rows Either a numeric vector of rows in the tabular object to be 
#'   modified or an object of class \code{call}.  When a \code{call}, 
#'   generated by \code{quote(expression)}, the expression resolves to 
#'   a logical vector the same length as the number of rows in the table.
#'   Sprinkles are applied to where the expression resolves to \code{TRUE}.
#' @param cols Either a numeric vector of columns in the tabular object to
#'   be modified, or a character vector of column names. A mixture of 
#'   character and numeric indices is permissible.
#' @param sanitize \code{logical(1)}. Should the code for the cell be sanitized.
#' @param sanitize_args A list of arguments to pass to 
#'   \code{Hmisc::latexTranslate}
#' @param part A character string denoting which part of the table to modify.
#' @param fixed \code{logical(1)} indicating if the values in \code{rows} 
#'   and \code{cols} should be read as fixed coordinate pairs.  By default, 
#'   sprinkles are applied at the intersection of \code{rows} and \code{cols}, 
#'   meaning that the arguments do not have to share the same length.  
#'   When \code{fixed = TRUE}, they must share the same length.
#' @param recycle A \code{character} one that determines how sprinkles are 
#'   managed when the sprinkle input doesn't match the length of the region
#'   to be sprinkled.  By default, recycling is turned off.  Recycling 
#'   may be performed across rows first (left to right, top to bottom), 
#'   or down columns first (top to bottom, left to right).
#' @param ... Additional arguments to pass to other methods. Currently ignored.
#' 
#' @details This sprinkle is only recognized by LaTeX output.  See 
#'   \code{\link[Hmisc]{latexTranslate}} for more details.
#'   
#' @section Functional Requirements:
#' \enumerate{
#'  \item Correctly reassigns the appropriate elements of \code{sanitize} 
#'    and \code{sanitize_args} columns in the table part.
#'  \item Casts an error if \code{x} is not a \code{dust} object.
#'  \item Casts an error if \code{sanitize} is not a \code{logical(1)}
#'  \item Casts an error if \code{sanitize_args} is not a \code{list}
#'  \item Casts an error if \code{part} is not one of \code{"body"}, 
#'    \code{"head"}, \code{"foot"}, or \code{"interfoot"}
#'  \item Casts an error if \code{fixed} is not a \code{logical(1)}
#'  \item Casts an error if \code{recycle} is not one of \code{"none"},
#'    \code{"rows"}, or \code{"cols"}
#' }
#' 
#' The functional behavior of the \code{fixed} and \code{recycle} arguments 
#' is not tested for this function. It is tested and validated in the
#' tests for \code{\link{index_to_sprinkle}}.
#' 
#' @seealso \code{\link{sprinkle}}, 
#'   \code{\link{index_to_sprinkle}}
#'   
#' @export

sprinkle_sanitize <- function(x, rows = NULL, cols = NULL, 
                           sanitize = NULL, sanitize_args = NULL,
                           part = c("body", "head", "foot", "interfoot"),
                           fixed = FALSE, 
                           recycle = c("none", "rows", "cols", "columns"),
                           ...)
{
  UseMethod("sprinkle_sanitize")
}

#' @rdname sprinkle_sanitize
#' @export

sprinkle_sanitize.dust <- function(x, rows = NULL, cols = NULL, 
                                sanitize = NULL, sanitize_args = NULL,
                                part = c("body", "head", "foot", "interfoot"),
                                fixed = FALSE, 
                                recycle = c("none", "rows", "cols", "columns"),
                                ...)
{
  coll <- checkmate::makeAssertCollection()
  
  if (!is.null(sanitize))
  {
    checkmate::assert_logical(x = sanitize,
                              len = 1,
                              add = coll)
  }
  
  if (!is.null(sanitize_args))
  {
    checkmate::assert_list(x = sanitize_args,
                           add = coll)
  }
  
  indices <- index_to_sprinkle(x = x, 
                               rows = rows, 
                               cols = cols, 
                               fixed = fixed,
                               part = part,
                               recycle = recycle,
                               coll = coll)
  
  checkmate::reportAssertions(coll)
  
  part <- part[1]
  
  if (!is.null(sanitize))
  {
    x[[part]][["sanitize"]][indices] <- sanitize
  }
  
  if (!is.null(sanitize_args))
  {
    x[[part]][["sanitize_args"]][indices] <- 
      paste0(
        deparse(sanitize_args),
        collapse = ""
      )
  }
  
  x
}

#' @rdname sprinkle_sanitize
#' @export

sprinkle_sanitize.dust_list <- function(x, rows = NULL, cols = NULL, 
                                     sanitize = NULL, sanitize_args = NULL,
                                     part = c("body", "head", "foot", "interfoot"),
                                     fixed = FALSE, 
                                     recycle = c("none", "rows", "cols", "columns"),
                                     ...)
{
  structure(
    lapply(X = x,
           FUN = sprinkle_sanitize.dust,
           rows = rows,
           cols = cols,
           sanitize = sanitize,
           sanitize_args = sanitize_args,
           part = part,
           fixed = fixed,
           recycle = recycle,
           ...),
    class = "dust_list"
  )
}

# Unexported Utility ------------------------------------------------

# These functions are to be used inside of the general `sprinkle` call
# When used inside `sprinkle`, the indices are already determined, 
# the only the `sanitize` and `sanitize_args` arguments needs to be validated. 
# The assert function is kept separate so it may be called earlier
# without attempting to perform the assignment.

sprinkle_align_index_assert <- function(halign, valign, coll)
{
  if (!is.null(sanitize))
  {
    checkmate::assert_logical(x = sanitize,
                              len = 1,
                              add = coll,
                              .var.name = "sanitize")
  }
  
  if (!is.null(sanitize_args))
  {
    checkmate::assert_list(x = sanitize_args,
                           add = coll,
                           .var.name = "sanitize_args")
  }
}

sprinkle_sanitize_index <- function(x, indices, sanitize, sanitize_args, part)
{
  if (!is.null(sanitize))
  {
    x[[part]][["sanitize"]][indices] <- sanitize
  }
  
  if (!is.null(sanitize_args))
  {
    x[[part]][["sanitize_args"]][indices] <- 
      paste0(
        deparse(sanitize_args),
        collapse = ""
      )
  }
  
  x
}