#' Build home section
#'
#' This function is responsible for building `.md` (or `.Rmd`) files typically
#' found in the root directory of the package. It will generate the home page
#' from  `index.Rmd` or `README.Rmd`, then `index.md` or `README.md`. If none of
#' these are found, it falls back to the description field in `DESCRIPTION`. It
#' also builds any files found in `.github/`.
#'
#' @section Images and figures:
#' If you want to include images in your `README.md`, they must be stored in
#' somewhere in the package so that they can be displayed on the CRAN website.
#' The best place to put them is `man/figures`. If you are generating figures
#' with R Markdown, make sure you set up `fig.path` as followed:
#'
#' \preformatted{
#' ```\{r, include = FALSE\}
#' knitr::opts_chunk$set(
#'   fig.path = "man/figures/"
#' )
#' ```
#' }
#'
#' @section Package logo:
#' If you have a package logo, you can include it at the top of your README in a
#' level-one heading:
#'
#' ```
#' # pkgdown <img src="man/figures/logo.png" align="right" />
#' ```
#'
#' [init_site()] will also automatically create a favicon set from your package
#' logo.
#'
#' @section YAML config - home:
#' To tweak the home page, you need a section called `home`.
#'
#' The sidebar links are automatically generated by inspecting the `URL` and
#' `BugReports` fields of the `DESCRIPTION`. You can add additional links with a
#' subsection called `links`, which should contain a list of `text` + `href`
#' elements:
#'
#' ```
#' home:
#'   links:
#'   - text: Link text
#'     href: http://website.com
#' ```
#'
#' You can remove the first heading with:
#'
#' ```
#' home:
#'   strip_header: true
#' ```
#'
#' @section YAML config - authors:
#' The "developers" list is populated by the maintainer ("cre"), authors
#' ("aut"), and funder ("fnd") from the `DESCRIPTION`. You can modify their
#' display on the home page by adding a subsection for `authors`. The author
#' name should match the value in `DESCRIPTION`:
#'
#' ```
#' authors:
#'   name:
#'     href: "http://name-website.com"
#'     html: "<img src='name-picture.png' height=24>"
#' ```
#'
#' @section ORCID:
#' ORCID identification numbers are added to author info if they
#' are defined in the `DESCRIPTION`:
#'
#' ```
#' person("Jay", "Hesselberth", role = "aut",
#'   comment = c(ORCID = "0000-0002-6299-179X")
#' ),
#' ```
#'
#' @section Badges:
#' Status badges are displayed in the sidebar under the section "Dev status".
#' This section is automatically populated if the first paragraph of the
#' homepage consists solely of status badges as linked images.
#'
#' @inheritParams build_articles
#' @export
build_home <- function(pkg = ".",
                       override = list(),
                       preview = NA,
                       quiet = TRUE) {

  pkg <- section_init(pkg, depth = 0L, override = override)
  rule("Building home")
  dir_create(pkg$dst_path)

  if (has_citation(pkg$src_path)) {
    build_citation_authors(pkg)
  } else {
    build_authors(pkg)
  }
  build_home_md(pkg)
  build_home_license(pkg)
  build_home_index(pkg, quiet = quiet)

  preview_site(pkg, "/", preview = preview)
}
