% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/object_ops.R
\name{export_report}
\alias{export_report}
\title{Export an \emph{agent}, \emph{informant}, \emph{multiagent}, or table scan to HTML}
\usage{
export_report(x, filename, path = NULL, quiet = FALSE)
}
\arguments{
\item{x}{An \emph{agent} object of class \code{ptblank_agent}, an \emph{informant} of class
\code{ptblank_informant}, a \emph{multiagent} of class \code{ptblank_multiagent}, a table
scan of class \code{ptblank_tbl_scan}, or, customized reporting objects
(\code{ptblank_agent_report}, \code{ptblank_informant_report},
\code{ptblank_multiagent_report.wide}, \code{ptblank_multiagent_report.long}).}

\item{filename}{The filename to create on disk for the HTML export of the
object provided. It's recommended that the extension \code{".html"} is included.}

\item{path}{An optional path to which the file should be saved (this is
automatically combined with \code{filename}).}

\item{quiet}{Should the function \emph{not} inform when the file is written? By
default this is \code{FALSE}.}
}
\value{
Invisibly returns \code{TRUE} if the file has been written.
}
\description{
The \emph{agent}, \emph{informant}, \emph{multiagent}, and the table scan object can be
easily written as HTML with \code{export_report()}. Furthermore, any report
objects from the \emph{agent}, \emph{informant}, and \emph{multiagent} (generated using
\code{\link[=get_agent_report]{get_agent_report()}}, \code{\link[=get_informant_report]{get_informant_report()}}, and
\code{\link[=get_multiagent_report]{get_multiagent_report()}}) can be provided here for HTML export. Each HTML
document written to disk is self-contained and easily viewable in a web
browser.
}
\section{Function ID}{

9-3
}

\examples{
if (interactive()) {

# A: Writing an agent report as HTML 

# Let's go through the process of (1)
# developing an agent with a validation
# plan (to be used for the data quality
# analysis of the `small_table` dataset),
# (2) interrogating the agent with the
# `interrogate()` function, and (3) writing
# the agent and all its intel to a file

# Creating an `action_levels` object is a
# common workflow step when creating a
# pointblank agent; we designate failure
# thresholds to the `warn`, `stop`, and
# `notify` states using `action_levels()`
al <- 
  action_levels(
    warn_at = 0.10,
    stop_at = 0.25,
    notify_at = 0.35
  )

# Now create a pointblank `agent` object
# and give it the `al` object (which
# serves as a default for all validation
# steps which can be overridden); the
# data will be referenced in `tbl`
agent <- 
  create_agent(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`export_report()`",
    actions = al
  )

# Then, as with any agent object, we
# can add steps to the validation plan by
# using as many validation functions as we
# want; then, we `interrogate()`
agent <-
  agent \%>\% 
  col_exists(vars(date, date_time)) \%>\%
  col_vals_regex(
    vars(b), regex = "[0-9]-[a-z]{3}-[0-9]{3}"
  ) \%>\%
  rows_distinct() \%>\%
  col_vals_gt(vars(d), value = 100) \%>\%
  col_vals_lte(vars(c), value = 5) \%>\%
  interrogate()

# The agent report can be written to an
# HTML file with `export_report()`
export_report(
  agent,
  filename = "agent-small_table.html"
)

# If you're consistently writing agent
# reports when periodically checking data,
# we could make use of `affix_date()` or
# `affix_datetime()` depending on the
# granularity you need; here's an example
# that writes the file with the format:
# 'agent-small_table-YYYY-mm-dd_HH-MM-SS.html'
export_report(
  agent,
  filename = affix_datetime(
    "agent-small_table.html"
  )
)

# B: Writing an informant report as HTML

# Let's go through the process of (1)
# creating an informant object that
# minimally describes the `small_table`
# dataset, (2) ensuring that data is
# captured from the target table using
# the `incorporate()` function, and (3)
# writing the informant report to HTML

# Create a pointblank `informant`
# object with `create_informant()`
# and the `small_table` dataset;
# `incorporate()` so that info snippets
# are integrated into the text
informant <- 
  create_informant(
    tbl = ~ small_table,
    tbl_name = "small_table",
    label = "`export_report()`"
  ) \%>\%
  info_snippet(
    snippet_name = "high_a",
    fn = snip_highest(column = "a")
  ) \%>\%
  info_snippet(
    snippet_name = "low_a",
    fn = snip_lowest(column = "a")
  ) \%>\%
  info_columns(
    columns = vars(a),
    info = "From {low_a} to {high_a}."
  ) \%>\%
  info_columns(
    columns = starts_with("date"),
    info = "Time-based values."
  ) \%>\%
  info_columns(
    columns = "date",
    info = "The date part of `date_time`."
  ) \%>\%
  incorporate()

# The informant report can be written
# to an HTML file with `export_report()`;
# let's do this with `affix_date()` so
# the filename has a datestamp
export_report(
  informant,
  filename = affix_date(
    "informant-small_table.html"
  )
)

# C: Writing a table scan as HTML

# We can get an report that describes all
# of the data in the `storms` dataset
tbl_scan <- scan_data(tbl = dplyr::storms)

# The table scan object can be written
# to an HTML file with `export_report()`
export_report(
  tbl_scan,
  filename = "tbl_scan-storms.html"
)

}

}
\seealso{
Other Object Ops: 
\code{\link{activate_steps}()},
\code{\link{deactivate_steps}()},
\code{\link{remove_steps}()},
\code{\link{set_tbl}()},
\code{\link{x_read_disk}()},
\code{\link{x_write_disk}()}
}
\concept{Object Ops}
