% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bake.R
\name{reproducibility tools}
\alias{reproducibility tools}
\alias{bake}
\alias{stew}
\alias{freeze}
\title{Tools for reproducible computations.}
\usage{
bake(
  file,
  expr,
  seed = NULL,
  kind = NULL,
  normal.kind = NULL,
  dependson = NULL,
  info = FALSE,
  timing = TRUE
)

stew(
  file,
  expr,
  seed = NULL,
  kind = NULL,
  normal.kind = NULL,
  dependson = NULL,
  info = FALSE
)

freeze(
  expr,
  seed = NULL,
  kind = NULL,
  normal.kind = NULL,
  envir = parent.frame(),
  enclos = if (is.list(envir) || is.pairlist(envir)) parent.frame() else baseenv()
)
}
\arguments{
\item{file}{Name of the binary data file in which the result will be stored or retrieved, as appropriate.
For \code{bake}, this will contain a single object and hence be an RDS file (extension \sQuote{rds});
for \code{stew}, this will contain one or more named objects and hence be an RDA file (extension \sQuote{rda}).}

\item{expr}{Expression to be evaluated.}

\item{seed, kind, normal.kind}{optional.
To set the state and of the RNG.
The default, \code{seed = NULL}, will not change the RNG state.
\code{seed} should be a single integer.
See \code{\link{set.seed}} for more information.}

\item{dependson}{arbitrary \R object (optional).
Variables on which the computation in \code{expr} depends.
A hash of these objects will be archived in \code{file}, along with the results of evaluation \code{expr}.
When \code{bake} or \code{stew} are called and \code{file} exists, the hash of these objects will be compared against the archived hash;
recomputation is forced when these do not match.
The dependencies should be specified as unquoted symbols:
use a list if there are multiple dependencies.}

\item{info}{logical.
If \code{TRUE}, the \dQuote{ingredients} of the calculation are returned as a list.
In the case of \code{bake}, this list is the \dQuote{ingredients} attribute of the returned object.
In the case of \code{stew}, this list is a hidden object named \dQuote{.ingredients}, located in the environment within which \code{stew} was called.}

\item{timing}{logical.
If \code{TRUE}, the time required for the computation is returned.
This is returned as the \dQuote{system.time} attribute of the returned object.}

\item{envir}{the \code{\link[base]{environment}} in which \code{expr} is to
    be evaluated.  May also be \code{NULL}, a list, a data frame,
    a pairlist or an integer as specified to \code{\link[base]{sys.call}}.}

\item{enclos}{Relevant when \code{envir} is a (pair)list or a data frame.
    Specifies the enclosure, i.e., where \R looks for objects not found
    in \code{envir}.  This can be \code{NULL} (interpreted as the base
    package environment, \code{\link[base]{baseenv}()}) or an environment.}
}
\value{
\code{bake} returns the value of the evaluated expression \code{expr}.
Other objects created in the evaluation of \code{expr} are discarded along with the temporary, local environment created for the evaluation.

The latter behavior differs from that of \code{stew}, which returns the names of the objects created during the evaluation of \code{expr}.
After \code{stew} completes, these objects are copied into the environment in which \code{stew} was called.

\code{freeze} returns the value of evaluated expression \code{expr}.
However, \code{freeze} evaluates \code{expr} within the parent environment, so other objects created in the evaluation of \code{expr} will therefore exist after \code{freeze} completes.

\code{bake} and \code{stew} store information about the code executed, the dependencies, and the state of the random-number generator in the archive file.
In the case of \code{bake}, this is recorded in the \dQuote{ingredients} attribute (\code{attr(.,"ingredients")});
in the \code{stew} case, this is recorded in an object, \dQuote{.ingredients}, in the archive.
This information is returned only if \code{info=TRUE}.

The time required for execution is also recorded.
\code{bake} stores this in the \dQuote{system.time} attribute of the archived \R object;
\code{stew} does so in a hidden variable named \code{.system.time}.
The timing is obtained using \code{\link{system.time}}.
}
\description{
Bake, stew, and freeze assist in the construction of reproducible computations.
}
\details{
On cooking shows, recipes requiring lengthy baking or stewing are prepared beforehand.
The \code{bake} and \code{stew} functions perform analogously:
an computation is performed and archived in a named file.
If the function is called again and the file is present, the computation is not executed.
Instead, the results are loaded from the archive.
Moreover, via their optional \code{seed} argument, \code{bake} and \code{stew} can control the pseudorandom-number generator (RNG) for greater reproducibility.
After the computation is finished, these functions restore the pre-existing RNG state to avoid side effects.

The \code{freeze} function doesn't save results, but does set the RNG state to the specified value and restore it after the computation is complete.

Both \code{bake} and \code{stew} first test to see whether \code{file} exists.
If it does, \code{bake} reads it using \code{\link{readRDS}} and returns the resulting object.
By contrast, \code{stew} loads the file using \code{\link{load}} and copies the objects it contains into the user's workspace (or the environment of the call to \code{stew}).

If \code{file} does not exist, then both \code{bake} and \code{stew} evaluate the expression \code{expr};
they differ in the results that they save.
\code{bake} saves the value of the evaluated expression to \code{file} as a single object.
The name of that object is not saved.
By contrast, \code{stew} creates a local environment within which \code{expr} is evaluated; all objects in that environment are saved (by name) in \code{file}.
\code{bake} and \code{stew} also store information about the code executed, the dependencies, and the state of the random-number generator (if the latter is controlled) in the archive file.
Re-computation is triggered if any of these things change.
}
\section{Compatibility with older versions}{

With \pkg{pomp} version 3.4.4.2, the behavior of \code{bake} and \code{stew} changed.
In particular, older versions did no dependency checking, and did not check to see whether \code{expr} had changed.
Accordingly, the archive files written by older versions have a format that is not compatible with the newer ones.
When an archive file in the old format is encountered, it will be updated to the new format, with a warning message.
\strong{Note that this will overwrite existing archive files!}
However, there will be no loss of information.
}

\examples{
\dontrun{
  bake(file="example1.rds",{
    x <- runif(1000)
    mean(x)
  })

  bake(file="example1.rds",{
    x <- runif(1000)
    mean(x)
  })

  bake(file="example1.rds",{
    a <- 3
    x <- runif(1000)
    mean(x)
  })

  a <- 5
  b <- 2

  stew(file="example2.rda",
    dependson=list(a,b),{
      x <- runif(10)
      y <- rnorm(n=10,mean=a*x+b,sd=2)
    })

  plot(x,y)

  set.seed(11)
  runif(2)
  freeze(runif(3),seed=5886730)
  runif(2)
  freeze(runif(3),seed=5886730)
  runif(2)

  set.seed(11)
  runif(2)
  runif(2)
  runif(2)

}
}
\author{
Aaron A. King
}
\concept{reproducibility}
