% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/n_eff.R
\name{n_eff}
\alias{n_eff}
\title{Calculates the effective sample size of the data}
\usage{
n_eff(
  kinship,
  max = TRUE,
  weights = NULL,
  nonneg = TRUE,
  algo = c("gradient", "newton", "heuristic"),
  tol = 1e-10
)
}
\arguments{
\item{kinship}{An \code{n}-by-\code{n} kinship matrix.}

\item{max}{If \code{TRUE}, returns the maximum \code{n_eff} value among those computed using all possible vectors of weights that sum to one (and which are additionally non-negative if \code{nonneg = TRUE}).
If \code{FALSE}, \code{n_eff} is computed using the specific weight vector provided.}

\item{weights}{Weights for individuals (optional).
If \code{NULL}, uniform weights are used.
This parameter is ignored if \code{max = TRUE}.}

\item{nonneg}{If \code{TRUE} (default) and \code{max = TRUE}, non-negative weights that maximize \code{n_eff} are found.
See \code{algo}.
This has no effect if \code{max = FALSE}.}

\item{algo}{Algorithm for finding optimal non-negative weights (applicable only if \code{nonneg = TRUE} and \code{max = TRUE} and the weights found by matrix inversion are non-negative).
May be abbreviated.
If "gradient" (default), an optimized gradient descent algorithm is used (fastest; recommended).
If "newton", the exact multivariate newton's Method is used (slowest since \code{(n+1)}-by-\code{(n+1)} Hessian matrix needs to be inverted at every iteration; use if possible to confirm that "gradient" gives the best answer).
If "heuristic", if the optimal solution by the inverse matrix method contains negative weights, the most negative weight in an iteration is forced to be zero in all subsequent iterations and the rest of the weights are solved for using the inverse matrix method, repeating until all resulting weights are non-negative (also slow, since inversion of large matrices is required; least likely to find optimal solution).}

\item{tol}{Tolerance parameter for "gradient" and "newton" algorithms.
The algorithms converge when the norm of the step vector is smaller than this tolerance value.}
}
\value{
A list containing \code{n_eff} and \code{weights} (optimal weights if \code{max = TRUE}, input weights otherwise).
}
\description{
The effective sample size \code{n_eff} is the equivalent number of independent haplotypes that gives the same variance as that observed under the given population.
The variance in question is for the weighted sample mean ancestral allele frequency estimator.
It follows that \code{n_eff} equals the inverse of the weighted mean kinship.
If \code{max = TRUE}, a calculation is performed that implicitly uses optimal weights which maximize \code{n_eff}, which equals the sum of the elements of the inverse kinship matrix.
However, if \code{nonneg = TRUE} and if the above solution has negative weights (common), optimal non-negative weights are found instead (there are three algorithms available, see \code{algo}).
If \code{max = FALSE}, then the input weights are used in this calculation, and if weights are \code{NULL}, uniform weights are used.
}
\details{
The maximum \code{n_eff} possible is \code{2 * n}, where \code{n} is the number of individuals; this value is attained only when all haplotypes are independent (a completely unstructured population in Hardy-Weinberg equilibrium).
The minimum \code{n_eff} possible is 1, which is attained in an extremely structured population with FST of 1, where every individual has exactly the same haplotype at every locus (no heterozygotes).
Moreover, for \code{K} extremely-differentiated subpopulations (FST = 1 per subpopulation) \code{n_eff = K}.
In this way, \code{n_eff} is smaller than the ideal value of \code{2 * n} depending on the amount of kinship (covariance) in the data.

Occasionally, depending on the quality of the input kinship matrix, the estimated \code{n_eff} may be outside the theoretical \verb{\[1, 2*n\]} range, in which case the return value is set to the closest boundary value.
The quality of the results depends on the success of matrix inversion (which for numerical reasons may incorrectly contain negative eigenvalues, for example) or of the gradient optimization.
}
\examples{
# Get n_eff from a genotype matrix

# Construct toy data
X <- matrix(c(0,1,2,1,0,1,1,0,2), nrow=3, byrow=TRUE) # genotype matrix
subpops <- c(1,1,2) # subpopulation assignments for individuals

# NOTE: for BED-formatted input, use BEDMatrix!
# "file" is path to BED file (excluding .bed extension)
## library(BEDMatrix)
## X <- BEDMatrix(file) # load genotype matrix object

# estimate the kinship matrix "kinship" from the genotypes "X"!
kinship <- popkin(X, subpops) # calculate kinship from X and optional subpop labels
weights <- weights_subpops(subpops) # can weigh individuals so subpopulations are balanced

# use kinship matrix to calculate n_eff
# default mode returns maximum n_eff possible across all non-negative weights that sum to one
# also returns the weights that were optimal
obj <- n_eff(kinship)
n_eff_max <- obj$n_eff
w_max <- obj$weights

# version that uses weights provided
obj <- n_eff(kinship, max = FALSE, weights = weights)
n_eff_w <- obj$n_eff
w <- obj$weights # returns input weights renormalized for good measure

# no (or NULL) weights implies uniform weights
obj <- n_eff(kinship, max = FALSE)
n_eff_u <- obj$n_eff
w <- obj$weights # uniform weights

}
