#' @title Compute weight and score matrices from PLSc
#' @description Compute weight and score matrices for Partial Least Squares classification
#' @usage computeWT(X, Y, A)
#' @param X Data matrix where columns represent the \eqn{p} variables and
#' rows the \eqn{n} observations.
#' @param Y Data matrix where columns represent the two classes and
#' rows the \eqn{n} observations.
#' @param A Number of score components
#' @importFrom nipals nipals
#' @author Angela Andreella
#' @return List with the following objects:
#' \describe{
#'   \item{W}{Matrix of weights}
#'   \item{T_score}{Matrix of \code{Y} scores}
#'   \item{R}{Matrix of \code{Y} residuals}
#' }
#' @export
#' @keywords internal



computeWT <- function(X, Y, A){



  E <- R <- w <- r <- Q <- list()

  E[[1]] <- X
  R[[1]] <- Y
  #number of observations
  N <- nrow(Y)

  #dim(W) = nX times A

  for(i in seq(A)){

    AA <- t(E[[i]]) %*% R[[i]] %*% t(R[[i]]) %*% E[[i]]

    #Compute weight matrix
    #out <- eigen(AA) #well defined eigenvalue problem
    out <- nipals(AA, center =FALSE, scale = FALSE)
   # w[[i+1]] <- Re(out$vectors[,1])
    w[[i+1]] <- Re(out$loadings[,1])

    #score vector t_i
    r[[i+1]] <- E[[i]] %*% w[[i+1]]

    #Orthogonal projection matrix of scores
    #Q[[i+1]] <- diag(N) - r[[i+1]] %*% t(r[[i+1]])/ (t(r[[i+1]]) %*% r[[i+1]])[1]
    Q[[i+1]] <- diag(N) - r[[i+1]] %*% solve(t(r[[i+1]]) %*% r[[i+1]]) %*% t(r[[i+1]])

    #Deflation step
    #X-deflation step
    E[[i+1]] <- Q[[i+1]] %*% E[[i]] #residual matrix X
    #Y-deflation step
    R[[i+1]] <- Q[[i+1]] %*% R[[i]] #residual matrix Y
  }

  #Rearrange weight matrix
  W <- NULL
  for (i in c(2:(length(w)))) W <- cbind(W, w[[i]]) #number of obs times A


  #Rearrange scores matrix
  T_score <- NULL
  for (i in c(2:(length(r)))) T_score <- cbind(T_score, r[[i]]) #number of obs times ncomponent


  return(list(W = W,
              T_score = T_score,
              R = R))
}
