% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crossv.R
\name{plot.precmed}
\alias{plot.precmed}
\title{Two side-by-side line plots of validation curves from the \code{"precmed"} object}
\usage{
\method{plot}{precmed}(
  x,
  cv.i = NULL,
  combine = "mean",
  show.abc = TRUE,
  valid.only = FALSE,
  plot.hr = FALSE,
  ylab = NULL,
  legend.position = "bottom",
  xlim = NULL,
  title = waiver(),
  theme = theme_classic(),
  ...
)
}
\arguments{
\item{x}{An object of class \code{"precmed"}.}

\item{cv.i}{A positive integer indicating the index of the CV iteration results to be plotted.
Allowed values are: a positive integer \eqn{<=} \code{cv.n} in \code{\link{catecv}()} or
\code{NULL}. If \code{cv.i = NULL}, the results across all CV iterations are combined according
to \code{combine} and then plotted. Default is \code{NULL}.}

\item{combine}{A character value indicating how to combine the estimated ATEs across all CV
iterations into a validation curve for each nested subgroup, separately for the training and
validation results. Allowed values are: \code{'mean'} or \code{'median'}. Used only if
\code{cv.i = NULL}. Default is \code{'mean'}.}

\item{show.abc}{A logical value indicating whether to show the ABC statistics in the validation set. Used
only if \code{x$abc = TRUE} and \code{xlim} is not limited to a smaller range (i.e., \code{xlim = NULL} or
equal to the entire \code{x$prop.onlyhigh} range). If \code{cv.i} is NULL, ABC statistics will be based
on the combined CV iterations. If \code{cv.i} is an integer, ABC statistics will be based solely on that
CV iteration. Default is \code{TRUE}.}

\item{valid.only}{A logical value indicating whether only the validation curves in the validation set
should be plotted (\code{TRUE}). Otherwise, the validation curves in both the training and validation
sets are plotted side-by-side (\code{FALSE}). Default is \code{FALSE}.}

\item{plot.hr}{A logical value indicating whether the hazard ratios should be plotted in the
validation curves (\code{TRUE}). Otherwise, the restricted mean time lost is plotted (\code{FALSE}).
This argument is only applicable to survival outcomes. Default is \code{FALSE}.}

\item{ylab}{A character value for the y-axis label to describe what the ATE is. Default is \code{NULL},
which creates a default y-axis label based on available data.}

\item{legend.position}{A character value for the legend position argument to be passed to \code{ggplot}
object. Default is \code{'bottom'}.}

\item{xlim}{A numeric value for the range of the x-axis. Default is \code{NULL}, which means there is no
range specified.}

\item{title}{The text for the title}

\item{theme}{Defaults to \code{theme_classic()}. Other options include \code{theme_grey()}, \code{theme_bw()}, \code{theme_light()}, \code{theme_dark()}, and \code{theme_void()}}

\item{...}{Other parameters}
}
\value{
Returns two side-by-side line plots, one of which shows the validation curves of the training
sets and the other the validation curves in the validation sets. A gray horizontal dashed line of
overall ATE is included as a reference. ABC statistics will be added to the legend if
\code{show.abc = TRUE}.
}
\description{
Provides validation curves in two side-by-side plots, visualizing the estimated ATEs in a series
of nested subgroups in the training set and validation set separately, where each line represents
one scoring method specified in \code{\link{catecv}()} or \code{\link{catecvmean}()}. This should be run
only after results of \code{\link{catecv}()} or \code{\link{catecvmean}()} have been obtained.
}
\details{
\code{\link{plot}()} takes in outputs from \code{\link{catecv}()} and generates two plots
of validation curves side-by-side, one for the training set and one for validation set.
Separate validation curves are produced for each scoring method specified via \code{score.method}
in \code{\link{catecv}()} or \code{\link{catecvmean}()}.

The validation curves (and ABC statistics, if applicable) can help compare the performance of
different scoring methods in terms of discerning potential treatment heterogeneity in subgroups
with internal validation. Steeper validation curves in the validation set suggest presence of
treatment effect heterogeneity (and the ability of the scoring methods to capture it) while flat
validation curves indicate absence of treatment effect heterogeneity (or inability of the scoring method
to capture it).
}
\examples{
\donttest{
# Count outcome
eval_1 <- catecv(response = "count",
                 data = countExample,
                 score.method = "poisson",
                 cate.model = y ~ age + female + previous_treatment +
                                  previous_cost + previous_number_relapses + offset(log(years)),
                 ps.model = trt ~ age + previous_treatment,
                 higher.y = FALSE,
                 cv.n = 5)

# default setting
plot(eval_1)

# turn off ABC annotation
plot(eval_1, show.abc = FALSE)

# use a different theme
plot(eval_1, theme = ggplot2::theme_bw())

# plot the validation curves from the 2nd CV iteration instead of the mean
# of all validation curves
plot(eval_1, cv.i = 2)

# median of the validation curves
plot(eval_1, combine = "median")

# plot validation curves in validation set only
plot(eval_1, valid.only = TRUE)

# Survival outcome
library(survival)
tau0 <- with(survivalExample,
             min(quantile(y[trt == "drug1"], 0.95), quantile(y[trt == "drug0"], 0.95)))
eval_2 <- catecv(response = "survival",
                 data = survivalExample,
                 score.method = c("poisson", "randomForest"),
                 cate.model = Surv(y, d) ~ age + female + previous_cost +
                                           previous_number_relapses,
                 ps.model = trt ~ age + previous_treatment,
                 initial.predictor.method = "randomForest",
                 ipcw.model = ~ age + previous_cost + previous_treatment,
                 tau0 = tau0,
                 cv.n = 5,
                 seed = 999)


# default setting, plot RMTL ratios in both training and validation sets
plot(eval_2)

# plot hazard ratio
plot(eval_2, plot.hr = TRUE)

}

}
\references{
Yadlowsky, S., Pellegrini, F., Lionetto, F., Braune, S., & Tian, L. (2020).
\emph{Estimation and validation of ratio-based conditional average treatment effects using
observational data. Journal of the American Statistical Association, 1-18.}
\url{https://www.tandfonline.com/doi/full/10.1080/01621459.2020.1772080}
}
\seealso{
\code{\link{abc}()} and \code{\link{boxplot}()} for \code{"precmed"} objects.
}
