#' @title Animate cases on a process map
#'
#' @description A function for creating a SVG animation of an event log on a process map created by processmapR.
#' @param eventlog The event log object that should be animated
#' @param processmap The process map created with processmapR that the event log should be animated on,
#'  if not provided a standard process map will be generated by using processmapR::process_map.
#' @param animation_mode Whether to animate the cases according to their actual time of occurence ("absolute") or to start all cases at once ("relative").
#' @param animation_duration The overall duration of the animation, all times are scaled according to this overall duration.
#' @param animation_jitter The magnitude of a random coordinate translation, known as jitter in scatterplots, which is added to each token. Adding jitter can help to disambiguate tokens traveling on top of each other.
#' @param animation_timeline Whether to render a timeline slider in supported browsers (Recent versions of Chrome and Firefox only).
#' @param token_size The event attribute (character) or alternatively a data frame with three columns (case, time, size) matching the case identifier of the supplied event log.
#'  The token size is scaled accordingly during the animation (default size is 4). You may use \code{\link{add_token_size}} to add a suitable attribute to the event log.
#' @param token_color The event attribute (character) or alternatively a data frame with three columns (case, time, color) matching the case identifier of the supplied event log.
#'  The token color is change accordingly during the animation (default color is orange). You may use \code{\link{add_token_color}} to add a suitable attribute to the event log.
#' @param token_image The event attribute (character) or alternatively a data frame with three columns (case, time, image) matching the case identifier of the supplied event log.
#'  The token image is change accordingly during the animation (by default a SVG shape is used).
#' @param token_opacity The event attribute (character) or alternatively a data frame with three columns (case, time, transparency) matching the case identifier of the supplied event log.
#'  The token fill-opacity is change accordingly during the animation (by default the token is dranw with 0.9 opacity).
#' @param token_shape The (fixed) SVG shape to be used to draw tokens. Can be either 'circle' (default) or 'rect'.
#' @param token_options A list of additional (fixed) SVG properties to be added to each token.
#' @param width,height Fixed size for widget (in css units). The default is NULL, which results in intelligent automatic sizing based on the widget's container.
#' @param ... Options passed on to \code{\link{process_map}}.
#'
#' @examples
#' # Load example event log
#' library(eventdataR)
#'
#' # Animate the process with default options (absolute time and 60s duration)
#' animate_process(patients)
#'
#' # Animate the process with default options (relative time and with jitter)
#' animate_process(patients, animation_mode = "relative", animation_jitter = 10)
#'
#' \donttest{
#'
#' #' # Change default token sizes
#' animate_process(patients, token_size = 2)
#'
#' # Change default token color
#' animate_process(patients, token_color = "red")
#'
#' # Change default token opacity
#' animate_process(patients, token_opacity = 0.5)
#'
#' # Change default token image (GIFs work too)
#' animate_process(patients, token_image = "https://upload.wikimedia.org/wikipedia/en/5/5f/Pacman.gif")
#'
#' # Change token color based on a numeric attribute, here the nonsensical 'time' of an event
#' animate_process(add_token_color(patients, "time", "color"), token_color = "color")
#'
#' # Change token color based on a factor attribute
#' animate_process(add_token_color(patients, "employee", "color",
#'                 color_mapping = scales::col_factor("Set3", patients$employee)),
#'                 token_color = "color")
#'
#' # Next example requires the 'dplyr' and 'edeaR' packages
#' library(dplyr)
#' library(edeaR)
#'
#' # Change token_color based on colors in a second data frame
#' # Extract only the lacticacid measurements
#' lactic <- sepsis %>%
#'     mutate(lacticacid = as.numeric(lacticacid)) %>%
#'     filter_activity(c("LacticAcid")) %>%
#'     as.data.frame() %>%
#'     select("case" = case_id, "time" =  timestamp, lacticacid)
#' # Create a numeric color scale
#' cscale <- scales::col_numeric("Oranges", lactic$lacticacid , na.color = "white")
#' # Create colors data frame for animate_process
#' lacticColors <- lactic %>%
#'     mutate(color = cscale(lacticacid))
#' sepsisBase <- sepsis %>%
#'     filter_activity(c("LacticAcid", "CRP", "Leucocytes", "Return ER",
#'                       "IV Liquid", "IV Antibiotics"), reverse = TRUE) %>%
#'     filter_trace_frequency(percentage = 0.95)
#' animate_process(sepsisBase, token_color = lacticColors, animation_mode = "relative",
#'                 animation_duration = 600)
#' }
#'
#'
#' @author Felix Mannhardt <felix.mannhardt@sintef.no> (SINTEF Digital)
#' @seealso processmapR:process_map
#'
#' @import dplyr
#' @import bupaR
#' @import processmapR
#' @importFrom magrittr %>%
#' @importFrom rlang :=
#' @importFrom rlang !!
#'
#' @export
animate_process <- function(eventlog,
                            processmap = process_map(eventlog, render = F, ...),
                            animation_mode = "absolute",
                            animation_duration = 60,
                            animation_jitter = 0,
                            animation_timeline = TRUE,
                            token_size = NULL,
                            token_color = NULL,
                            token_image = NULL,
                            token_opacity = NULL,
                            token_shape = c("circle","rect"),
                            token_options = NULL,
                            width = NULL,
                            height = NULL,
                            ...) {

  # Make CRAN happy about dplyr evaluation
  case_start <- log_end <- start_time <- end_time <- next_end_time <- next_start_time <- NULL
  case <- case_end <- log_start <- log_duration <- case_duration <- NULL
  from_id <- to_id <- NULL
  label <- NULL

  token_shape <- match.arg(token_shape)

  # Generate the DOT source
  graph <- DiagrammeR::render_graph(processmap, width = width, height = height)
  # Get the DOT source for later rendering through vis.js
  diagram <- graph$x$diagram

  precedence <- attr(processmap, "base_precedence") %>%
    mutate_at(vars(start_time, end_time, next_start_time, next_end_time), as.numeric, units = "secs")

  cases <- precedence %>%
    group_by(case) %>%
    filter(!is.na(case)) %>%
    summarise(case_start = min(start_time, na.rm = T),
              case_end = max(end_time, na.rm = T)) %>%
    mutate(case_duration = case_end - case_start) %>%
    ungroup()

  # determine animation factor based on requested duration
  if (animation_mode == "absolute") {
    timeline_start <- cases %>% pull(case_start) %>% min(na.rm = T)
    timeline_end <- cases %>% pull(case_end) %>% max()
    animation_factor <- (timeline_end - timeline_start) / animation_duration
  } else {
    timeline_start <- 0
    timeline_end <- cases %>% pull(case_duration) %>% max(na.rm = T)
    animation_factor =  timeline_end / animation_duration
  }

  sizes <- generate_animation_attribute(eventlog, "size", token_size, 6)
  sizes <- transform_time(sizes, "size", cases, animation_mode, animation_factor, timeline_start, timeline_end)

  colors <- generate_animation_attribute(eventlog, "color", token_color, "white")
  colors <- transform_time(colors, "color", cases, animation_mode, animation_factor, timeline_start, timeline_end)

  images <- generate_animation_attribute(eventlog, "image", token_image, NA)
  images <- transform_time(images, "image", cases, animation_mode, animation_factor, timeline_start, timeline_end)

  opacities <- generate_animation_attribute(eventlog, "opacity", token_opacity, 0.9)
  opacities <- transform_time(opacities, "opacity", cases, animation_mode, animation_factor, timeline_start, timeline_end)

  tokens <- generate_tokens(cases, precedence, processmap, animation_mode, animation_factor, timeline_start, timeline_end)
  start_activity <- processmap$nodes_df %>% filter(label == "Start") %>% pull(id)
  end_activity <- processmap$nodes_df %>% filter(label == "End") %>% pull(id)
  case_ids <- tokens %>% distinct(case) %>% pull(case)

  settings <- list()
  x <- list(
    diagram = diagram,
    tokens = tokens,
    sizes = sizes,
    colors = colors,
    opacities = opacities,
    options = token_options,
    cases = case_ids,
    images = images,
    shape = token_shape,
    start_activity = start_activity,
    end_activity = end_activity,
    duration = animation_duration,
    timeline = animation_timeline,
    mode = animation_mode,
    jitter = animation_jitter,
    factor = animation_factor * 1000,
    timeline_start = timeline_start * 1000,
    timeline_end = timeline_end * 1000
  )

  htmlwidgets::createWidget(name = "processanimateR", x = x,
                            width = width, height = height,
                            sizingPolicy = htmlwidgets::sizingPolicy(
                              defaultWidth = 800,
                              defaultHeight = 600,
                              browser.fill = TRUE
                            ))
}

#' @title Create a process animation output element
#' @description Renders a renderProcessanimater within an application page.
#' @param outputId Output variable to read the animation from
#' @param width,height Must be a valid CSS unit (like 100%, 400px, auto) or a number,
#'  which will be coerced to a string and have px appended.
#'
#' @export
processanimaterOutput <- function(outputId, width = "100%", height = "400px") {
  htmlwidgets::shinyWidgetOutput(outputId = outputId,
                                 name = "processanimateR",
                                 inline = F,
                                 width = width, height = height,
                                 package = "processanimateR")
}

#' @title Renders process animation output
#' @description Renders a SVG process animation suitable to be used by processanimaterOutput.
#' @param expr The expression generating a process animation (animate_process).
#' @param env The environment in which to evaluate expr.
#' @param quoted Is expr a quoted expression (with quote())? This is useful if you want to save an expression in a variable.
#'
#' @export
renderProcessanimater <- function(expr, env = parent.frame(), quoted = FALSE) {
  if (!quoted) { expr <- substitute(expr) } # force quoted
  htmlwidgets::shinyRenderWidget(expr, processanimaterOutput, env, quoted = TRUE)
}

#
# Private helper functions
#

generate_tokens <- function(cases, precedence, processmap, animation_mode, animation_factor, timeline_start, timeline_end) {

  case <- end_time <- start_time <- next_end_time <- next_start_time <- case_start <- token_duration <- NULL
  min_order <- token_start <- activity_duration <- token_end <- from_id <- to_id <- case_duration <- NULL

  tokens <- precedence %>%
    left_join(cases, by = c("case")) %>%
    left_join(processmap$edges_df, by = c("from_id" = "from", "to_id" = "to")) %>%
    filter(!is.na(id) & !is.na(case))

  # SVG animations seem to not like events starting at the same time caused by 0s durations
  EPSILON = 0.00001

  if (animation_mode == "absolute") {
    tokens <- mutate(tokens,
                     token_start = (end_time - timeline_start) / animation_factor,
                     token_duration = (next_start_time - end_time) / animation_factor,
                     activity_duration = EPSILON + pmax(0, (next_end_time - next_start_time) / animation_factor))
  } else {
    tokens <- mutate(tokens,
                     token_start = (end_time - case_start) / animation_factor,
                     token_duration = (next_start_time - end_time) / animation_factor,
                     activity_duration = EPSILON + pmax(0, (next_end_time - next_start_time) / animation_factor))
  }

  tokens <- tokens %>%
    # Filter all negative durations caused by parallelism (TODO, deal with it in a better way)
    # Also, SMIL does not like 0 duration animateMotion
    filter(token_duration >= 0, activity_duration >= 0) %>%
    group_by(case) %>%
    # Ensure start times are not overlapping SMIL does not fancy this
    arrange(start_time, min_order) %>%
    # Add small delta for activities with same start time
    mutate(token_start = token_start + ((row_number(token_start) - min_rank(token_start)) * EPSILON)) %>%
    # Ensure consecutive start times
    mutate(token_end = min(token_start) + cumsum(token_duration + activity_duration) + EPSILON) %>%
    mutate(token_start = lag(token_end, default = min(token_start))) %>%
    # Adjust case duration
    mutate(case_duration = max(token_end)) %>%
    ungroup()

  tokens %>%
    select(case,
           edge_id = id,
           from_id,
           to_id,
           token_start,
           token_duration,
           activity_duration,
           case_duration)

}

generate_animation_attribute <- function(eventlog, attributeName, value, default) {
  attribute <- rlang::sym(attributeName)
  # standard token size
  if (is.null(value)) {
    eventlog %>%
      as.data.frame() %>%
      select(case = !!case_id_(eventlog),
             time = !!timestamp_(eventlog)) %>%
      mutate(!!attribute := default)
  } else if (is.data.frame(value)) {
    stopifnot(c("case", "time", attributeName) %in% colnames(value))
    value
  } else if (value %in% colnames(eventlog)) {
    eventlog %>%
      as.data.frame() %>%
      select(case = !!case_id_(eventlog),
             time = !!timestamp_(eventlog),
             !!rlang::sym(value)) %>%
      mutate(!!attribute := !!rlang::sym(value))
  } else {
    eventlog %>%
      as.data.frame() %>%
      select(case = !!case_id_(eventlog),
             time = !!timestamp_(eventlog)) %>%
      mutate(!!attribute := value)
  }
}

transform_time <- function(data, col, cases, animation_mode, animation_factor, timeline_start, timeline_end) {

  .order <- time <- case <- log_start <- case_start <- NULL

  col <- rlang::sym(col)
  data <- data %>%
    group_by(case) %>%
    filter(row_number(!!col) == 1 | lag(!!col) != !!col) %>%
    left_join(cases, by = "case")

  if (animation_mode == "absolute") {
    data <- data %>%
      mutate(time = as.numeric(time - timeline_start, units = "secs") / animation_factor) %>%
      select(case, time, !!col)
  } else {
    col <- data %>%
      mutate(time = as.numeric(time - case_start, units = "secs") / animation_factor) %>%
      select(case, time, !!col)
  }

}

# Utility functions
# https://github.com/gertjanssenswillen/processmapR/blob/master/R/utils.R
case_id_ <- function(eventlog) rlang::sym(case_id(eventlog))
timestamp_ <- function(eventlog) rlang::sym(timestamp(eventlog))
