
#' Calculate the relative infectivity of infectious individuals
#'
#' This function calculates the relative infectivity of infectious indiviuals at
#' time \eqn{t+1}, generated by cases with onset at time 1, 2, ..., t. The
#' relative infectivity of an individual on a given day is defined as the
#' probability that new infections generated by this individual show symptoms on
#' that day, and is defined by the serial interval distribution. The function
#' sums these values across all infected individuals (see details for more
#' information). Calculations use matrices where independent simulations are
#' stored as separate columns.
#'
#' @param w a `numeric` containing numbers representing the PMF of the serial
#'   interval, starting at day 1, i.e. one day after symptom onset; the vector
#'   should always be at least as long as the largest value of `t` allowed
#'
#' @param cases a `matrix` of `integers` with `t` rows (each row is a date) and
#'   `n_sim` columns (each column is a separate simulation) indicating number of
#'   cases on a given day, in a given simulation
#'
#' @param t an `integer` indicating the simulation step: incidence will then be
#'   computed for `t+1` taking into account past cases and R from time point `1`
#'   until `t`
#'
#' @seealso `compute_force_infection` does a similar
#'   thing, but also factors in the reproduction numbers of all individuals,
#'   resulting in an estimate of average number of new cases at `t+1`.
#'
#' @author Thibaut Jombart, with inputs from Anne Cori and Pierre Nouvellet
#' 
#' @details
#'
#' The relative infectivity for \eqn{i=1,...n} individuals at time \eqn{t+1} is
#'   defined as:
#' 
#' \deqn{ \sum_{i = 1}^n w(t - t_i) }
#'
#' where \eqn{w(.)} is the PMF of the serial interval distribution and \eqn{t_i} is the
#' date of onset of case *i*.
#'
#' This is equivalent to:
#'
#' \deqn{
#'   \sum_{s=1}^t y_s w(s - t_i)
#' }
#' 
#' where \eqn{y_s} is the number of new cases at time *s*. This latter
#' formulation is the one used in the function.
#'
#' @noRd

compute_relative_infectivity <- function(w, cases, t) {
  rev_w <- rev(w)
  ws <- utils::tail(rev_w, t)
  cases <- cases[seq_len(t), , drop = FALSE]
  out <- ws %*% cases
  out
}
