% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/r_contin_table_zip.R
\name{r_contin_table_zip}
\alias{r_contin_table_zip}
\title{Generate random contingency tables for adverse event
(across rows) and drug (across columns) combinations
given row and column marginal totals, embedded signals,
and possibly with zero inflation}
\usage{
r_contin_table_zip(
  n = 1,
  row_marginals,
  col_marginals,
  signal_mat = matrix(1, length(row_marginals), length(col_marginals)),
  omega_vec = rep(0, length(col_marginals)),
  no_zi_idx = NULL,
  ...
)
}
\arguments{
\item{n}{number of random matrices to generate.}

\item{row_marginals, col_marginals}{(possibly named) vector of row and column
marginal totals. Must add up to the same total. If named, the names are passed
on to the randomly generated matrix/matrices.}

\item{signal_mat}{numeric matrix of dimension
length(row_marginals) x length(col_marginals). The (i, j)-th entry of
signal_mat determines the signal strength of the i-th adverse event and
j-th drug pair. The default is 1 for each pair, which means no signal for the pair.}

\item{omega_vec}{vector of zero inflation probabilities. Must be of the same length
as col_marginals.}

\item{no_zi_idx}{List of pairs {(i, j)} where zero inflation is not allowed. To
specify the entirety i-th row (or j-th column) use c(i, 0) (or c(0, j)). See examples.}

\item{...}{Additional arguments. Currently unused.}
}
\value{
A list of length \code{n}, with each entry being a
\code{length(row_marginal)} by \code{length(col_marginal)} matrix.
}
\description{
Generate random contingency tables for adverse event
(across rows) and drug (across columns) combinations
given row and column marginal totals, embedded signals,
and possibly with zero inflation
}
\examples{

set.seed(42)

# first load the 46 statin data
data(statin46)
# zero inflation probabilities
omega_tru <- c(rep(0.15, ncol(statin46) - 1), 0)

# signal matrix
signal_mat <- matrix(1, nrow(statin46), ncol(statin46))

# "positive" signal at the (1, 1) entry
# the first column
signal_mat[1, 1] <- 10

# Now simulate data with the same row/column marginals
# as in statin46, with embedded signals as specified in
# the above signal_mat

# no zero inflation at (1, 1)
# (where signal is elicited) and the last row ("Other PT")
# and at the last column ("Other drugs") of the simulated matrix

sim_statin <- r_contin_table_zip(
  n = 1,
  row_marginals = rowSums(statin46),
  col_marginals = colSums(statin46),
  signal_mat = signal_mat,
  omega_vec = omega_tru,
  no_zi_idx = list(
    c(1, 1),
    c(nrow(statin46), 0), # the entire last row
    c(0, ncol(statin46)) # the entire last column
  )
)[[1]]

# now analyze the above simulated data

# using a pseudo LRT with a ZIP model
test1 <- pvlrt(
  contin_table = sim_statin,
  nsim = 500
  # set to 500 for demonstration purposes only,
  # we recommend the default 10000 or bigger
)
extract_zi_probability(test1)
extract_significant_pairs(test1)

# LRT with a poisson model
test2 <- lrt_poisson(
  contin_table = sim_statin,
  nsim = 500
  # set to 500 for demonstration purposes only,
  # we recommend the default 10000 or bigger
)
extract_zi_probability(test2)
extract_significant_pairs(test2)

# LRT with true omega supplied
test3 <- pvlrt(
  contin_table = sim_statin,
  zi_prob = omega_tru,
  nsim = 500
  # set to 500 for demonstration purposes only,
  # we recommend the default 10000 or bigger
)
extract_zi_probability(test3)
extract_significant_pairs(test3)

}
