% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat_qq_band.R
\name{stat_qq_band}
\alias{stat_qq_band}
\title{Quantile-quantile confidence bands}
\usage{
stat_qq_band(data = NULL, mapping = NULL, geom = "ribbon",
  position = "identity", show.legend = NA, inherit.aes = TRUE,
  na.rm = TRUE, distribution = "norm", dparams = list(),
  bandType = "normal", B = 1000, conf = 0.95, mu = NULL, sigma = NULL,
  detrend = FALSE, ...)
}
\arguments{
\item{data}{The data to be displayed in this layer. There are three
   options:

   If \code{NULL}, the default, the data is inherited from the plot
   data as specified in the call to \code{\link{ggplot}}.

   A \code{data.frame}, or other object, will override the plot
   data. All objects will be fortified to produce a data frame. See
   \code{\link{fortify}} for which variables will be created.

   A \code{function} will be called with a single argument,
   the plot data. The return value must be a \code{data.frame.}, and
   will be used as the layer data.}

\item{mapping}{Set of aesthetic mappings created by \code{\link{aes}} or
\code{\link{aes_}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{geom}{The geometric object to use display the data}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link{borders}}.}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{distribution}{Character. Theoretical probability distribution function
to use. Do not provide the full distribution function name (e.g.,
\code{"dnorm"}). Instead, just provide its shortened name (e.g.,
\code{"norm"}). If you wish to provide a custom distribution, you may do so
by first creating the density, quantile, and random functions following the
standard nomenclature from the \code{stats} package (i.e., for
\code{"custom"}, create the \code{"dcustom"}, \code{"qcustom"}, and
\code{"rcustom"} functions).}

\item{dparams}{List of additional parameters passed on to the previously
chosen \code{distribution} function.}

\item{bandType}{Character. Either \code{"normal"}, \code{"bs"} or
\code{"ts"}. \code{"normal"} constructs simultaneous confidence bands based
on Normal confidence intervals. \code{"bs"} creates pointwise confidence
bands based on a parametric bootstrap; parameters are estimated with MLEs.
Finally, \code{"ts"} constructs tail-sensitive confidence bands, as
described by Aldor-Noiman et al. (2013) (also, see 'Note' for
limitations).}

\item{B}{Integer. If \code{bandType = "bs"}, then \code{B} is the number of
bootstrap replicates. If \code{bandType = "ts"}, then \code{B} is the
number of simulated samples.}

\item{conf}{Numerical. Confidence level of the bands.}

\item{mu}{Numerical. Only used if \code{bandType = "ts"}. Center
distributional parameter used to construct the simulated tail-sensitive
confidence bands. If either \code{mu} or \code{sigma} are \code{NULL}, then
those parameters are estimated using \code{\link[robustbase]{Qn}} and
\code{\link[robustbase]{s_Qn}}, respectively.}

\item{sigma}{Numerical. Only used if \code{bandType = "ts"}. Scale
distributional parameter used to construct the simulated tail-sensitive
confidence bands. If either \code{mu} or \code{sigma} are \code{NULL}, then
those parameters are estimated using \code{\link[robustbase]{Qn}} and
\code{\link[robustbase]{s_Qn}}, respectively.}

\item{detrend}{Logical. Should the plot objects be detrended? If \code{TRUE},
the objects will be detrended according to the reference Q-Q line. This
procedure was described by Thode (2002), and may help reducing visual bias
caused by the orthogonal distances from Q-Q points to the reference line.}

\item{...}{other arguments passed on to \code{\link{layer}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{color = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}
}
\description{
Draws quantile-quantile confidence bands, with an additional detrend option.
}
\note{
\itemize{
\item{Tail-sensitive confidence bands are only implemented for Normal Q-Q
plots. As a future update, we intend to generalize to other distributions.}
\item{Bootstrap bands are constructed based on a MLE parametric bootstrap.
Hence, it is not possible to construct such bands if the sample and
theoretical distributions present mismatching supports.}
}
}
\examples{
# generate random Normal data
set.seed(0)
smp <- data.frame(norm = rnorm(100))

# Normal Q-Q plot of Normal data
gg <- ggplot(data = smp, mapping = aes(sample = norm)) +
 stat_qq_band() +
 stat_qq_line() +
 stat_qq_point()
gg + labs(x = "Theoretical Quantiles", y = "Sample Quantiles")

# Exponential Q-Q plot of mean ozone levels (airquality dataset)
di <- "exp"
dp <- list(rate = 1)
gg <- ggplot(data = airquality, mapping = aes(sample = Ozone)) +
 stat_qq_band(distribution = di, dparams = dp) +
 stat_qq_line(distribution = di, dparams = dp) +
 stat_qq_point(distribution = di, dparams = dp) +
 labs(x = "Theoretical Quantiles", y = "Sample Quantiles")
gg

# Detrended Exponential Q-Q plot of mean ozone levels
di <- "exp"
dp <- list(rate = 1)
de <- TRUE
gg <- ggplot(data = airquality, mapping = aes(sample = Ozone)) +
 stat_qq_band(distribution = di, detrend = de) +
 stat_qq_line(distribution = di, detrend = de) +
 stat_qq_point(distribution = di, detrend = de) +
 labs(x = "Theoretical Quantiles", y = "Sample Quantiles")
gg

# Normal Q-Q plot of Normal data with boostrap confidence bands
bt <- "bs"
gg <- ggplot(data = smp, mapping = aes(sample = norm)) +
 stat_qq_band(bandType = bt) +
 stat_qq_line() +
 stat_qq_point() +
 labs(x = "Theoretical Quantiles", y = "Sample Quantiles")
gg

# Normal Q-Q plot of Normal data with tail-sensitive confidence bands
bt <- "ts"
gg <- ggplot(data = smp, mapping = aes(sample = norm)) +
 stat_qq_band(bandType = bt) +
 stat_qq_line() +
 stat_qq_point() +
 labs(x = "Theoretical Quantiles", y = "Sample Quantiles")
gg

}
\references{
\itemize{
\item{\href{https://www.crcpress.com/Testing-For-Normality/Thode/p/book/9780824796136}{Thode,
H. (2002), Testing for Normality. CRC Press, 1st Ed.}}
\item{\href{http://www.tandfonline.com/doi/abs/10.1080/00031305.2013.847865}{Aldor-Noiman,
S. et al. (2013). The Power to See: A New Graphical Test of Normality. The
American Statistician. 67:4.}}
}
}
