% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lazyload_cache.R
\name{lazyload_cache_dir}
\alias{lazyload_cache_dir}
\alias{lazyload_cache_labels}
\title{Lazyload Cache}
\usage{
lazyload_cache_dir(
  path = "./cache",
  envir = parent.frame(),
  ask = FALSE,
  verbose = TRUE,
  full.names = TRUE,
  ...
)

lazyload_cache_labels(
  labels,
  path = "./cache/",
  envir = parent.frame(),
  verbose = TRUE,
  filter,
  full.names = TRUE,
  ...
)
}
\arguments{
\item{path}{the path to the cache directory.}

\item{envir}{the environment to load the objects into}

\item{ask}{if TRUE ask the user to confirm loading each database found in
\code{path}}

\item{verbose}{if TRUE display the chunk labels being loaded}

\item{full.names}{use the full name, i.e., include the path, for the chunk
label? This argument is passed to \code{\link[base]{list.files}}.}

\item{...}{additional arguments passed to \code{\link[base]{list.files}}.}

\item{labels}{a character vector of the chunk labels to load.}

\item{filter}{an optional function passed to \code{\link[base:lazyload]{lazyLoad}}.
when called on a character vector of object names returns a logical vector:
only objects for which this is true will be loaded.}
}
\description{
Lazyload Cached label(s) or a whole directory.
}
\details{
These functions helpful for loading cached chunks into an interactive R
session.  Consider the following scenario: you use knitr and have cached
chunks for lazyloading.  You've created the document, close up your IDE and
move on to the next project.  Later, you revisit the initial project and need
to retrieve the objects created in the cached chunks.  One option is to
reevaluate all the code, but this could be time consuming.  The other option
is to use \code{lazyload_cache_labels} or \code{lazyload_cache_dir} to
quickly (lazy)load the chunks into an active R session.

Use \code{lazyload_cache_dir} to load a whole directory of cached objects.

Use \code{lazyload_cache_labels} to load and explicit set of cached chunks.
}
\examples{
# this example is based on \url{https://stackoverflow.com/a/41439691/1104685}

# create a temp directory for a and place a .Rmd file within
tmpdr <- paste0(tempdir(), "/llcache_eg")
dir.create(tmpdr)
old_wd <- getwd()
setwd(tmpdr)

cat("---",
    "title: \"A Report\"",
    "output: html_document",
    "---",
    "",
    "```{r first-chunk, cache = TRUE}",
    "mpg_by_wt_hp <- lm(mpg ~ wt + hp, data = mtcars)",
    "x_is_pi <- pi",
    "```",
    "",
    "```{r second-chunk, cache = TRUE}",
    "mpg_by_wt_hp_am <- lm(mpg ~ wt + hp + am, data = mtcars)",
    "x_is_e <- exp(1)",
    "```",
    sep = "\n",
    file = paste0(tmpdr, "/report.Rmd"))

# knit the file.  evaluate the chuncks in a new environment so we can compare
# the objects after loading the cache.
kenv <- new.env()
knitr::knit(input = paste0(tmpdr, "/report.Rmd"), envir = kenv)

# The objects defined in the .Rmd file are now in kenv
ls(envir = kenv)

# view the cache
list.files(path = tmpdr, recursive = TRUE)

# create another environment, and load only the second chunk
lenv <- new.env()
ls(envir = lenv)

lazyload_cache_labels(labels = "second-chunk",
                      path = paste0(tmpdr, "/cache"),
                      envir = lenv)
lenv$x_is_e
lenv$mpg_by_wt_hp_am

# load all the chuncks
menv <- new.env()
lazyload_cache_dir(path = paste0(tmpdr, "/cache"), envir = menv)

ls(envir = menv)
menv$x_is_pi
menv$x_is_e

# cleanup
setwd(old_wd)
unlink(tmpdr, recursive = TRUE)

}
