% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse.R
\name{analyse}
\alias{analyse}
\title{Analyse Multiple Imputed Datasets}
\usage{
analyse(imputations, fun = ancova, delta = NULL, ...)
}
\arguments{
\item{imputations}{An \code{imputations} object as created by \code{\link[=impute]{impute()}}.}

\item{fun}{An analysis function to be applied to each imputed dataset. See details.}

\item{delta}{A \code{data.frame} containing the delta transformation to be applied to the imputed
datasets prior to running \code{fun}. See details.}

\item{...}{Additional arguments passed onto \code{fun}.}
}
\description{
This function takes multiple imputed datasets (as generated by
the \code{\link[=impute]{impute()}} function) and runs an analysis function on
each of them.
}
\details{
This function works by performing the following steps:
\enumerate{
\item Extract a dataset from the \code{imputations} object.
\item Apply any delta adjustments as specified by the \code{delta} argument.
\item Run the analysis function \code{fun} on the dataset.
\item Repeat steps 1-3 across all of the datasets inside the \code{imputations}
object.
\item Collect and return all of the analysis results.
}

The analysis function \code{fun} must take a \code{data.frame} as its first
argument. All other options to \code{\link[=analyse]{analyse()}} are passed onto \code{fun}
via \code{...}.
\code{fun} must return a named list with each element itself being a
list containing a single
numeric element called \code{est} (or additionally \code{se} and \code{df} if
you had originally specified \code{\link[=method_bayes]{method_bayes()}} or \code{\link[=method_approxbayes]{method_approxbayes()}})
i.e.:\preformatted{myfun <- function(dat, ...) \{
    mod_1 <- lm(data = dat, outcome ~ group)
    mod_2 <- lm(data = dat, outcome ~ group + covar)
    x <- list(
        trt_1 = list(
            est = coef(mod_1)[[group]],
            se = sqrt(vcov(mod_1)[group, group]),
            df = df.residual(mod_1)
        ),
        trt_2 = list(
            est = coef(mod_2)[[group]],
            se = sqrt(vcov(mod_2)[group, group]),
            df = df.residual(mod_2)
        )
     )
     return(x)
 \}
}

Please note that the \code{vars$subjid} column (as defined in the original call to
\code{\link[=draws]{draws()}}) will be scrambled in the data.frames that are provided to \code{fun}.
This is to say they will not contain the original subject values and as such
any hard coding of subject ids is strictly to be avoided.

By default \code{fun} is the \code{\link[=ancova]{ancova()}} function.
Please note that this function
requires that a \code{vars} object, as created by \code{\link[=set_vars]{set_vars()}}, is provided via
the \code{vars} argument e.g. \code{analyse(imputeObj, vars = set_vars(...))}. Please
see the documentation for \code{\link[=ancova]{ancova()}} for full details.
Please also note that the theoretical justification for the conditional mean imputation
method (\code{method = method_condmean()} in \code{\link[=draws]{draws()}}) relies on the fact that ANCOVA is
a linear transformation of the outcomes.
Thus care is required when applying alternative analysis functions in this setting.

The \code{delta} argument can be used to specify offsets to be applied
to the outcome variable in the imputed datasets prior to the analysis.
This is typically used for sensitivity or tipping point analyses. The
delta dataset must contain columns \code{vars$subjid}, \code{vars$visit} (as specified
in the original call to \code{\link[=draws]{draws()}}) and \code{delta}. Essentially this \code{data.frame}
is merged onto the imputed dataset by \code{vars$subjid} and \code{vars$visit} and then
the outcome variable is modified by:\preformatted{imputed_data[[vars$outcome]] <- imputed_data[[vars$outcome]] + imputed_data[["delta"]]
}

Please note that in order to provide maximum flexibility, the \code{delta} argument
can be used to modify any/all outcome values including those that were not
imputed. Care must be taken when defining offsets. It is recommend that you
use the helper function \code{\link[=delta_template]{delta_template()}} to define the delta datasets as
this provides utility variables such as \code{is_missing} which can be used to identify
exactly which visits have been imputed.
}
\examples{
\dontrun{
vars <- set_vars(
    subjid = "subjid",
    visit = "visit",
    outcome = "outcome",
    group = "group",
    covariates = c("sex", "age", "sex*age")
)

analyse(
    imputations = imputeObj,
    vars = vars
)

deltadf <- data.frame(
    subjid = c("Pt1", "Pt1", "Pt2"),
    visit = c("Visit_1", "Visit_2", "Visit_2"),
    delta = c( 5, 9, -10)
)

analyse(
    imputations = imputeObj,
    delta = deltadf,
    vars = vars
)
}
}
\seealso{
\code{\link[=extract_imputed_dfs]{extract_imputed_dfs()}} for manually extracting imputed
datasets.

\code{\link[=delta_template]{delta_template()}} for creating delta data.frames.

\code{\link[=ancova]{ancova()}} for the default analysis function.
}
