#' Process tracing estimator
#'
#' Draw conclusions from a model given a query, data, and process tracing strategies
#'
#' See https://book.declaredesign.org/observational-causal.html#process-tracing
#'
#' @param causal_model a model generated by `CausalQueries`
#' @param data a single row dataset with data on nodes in the model
#' @param query a causal query of interest
#' @param strategies a vector describing sets of nodes to be examined e.g. c("X", "X-Y")
#'
#' @return a data.frame of estimates
#'
#' @export
#' @examples
#'
#' process_tracing_estimator(
#'   CausalQueries::make_model("X -> Y"),
#'   "Y[X=1] > Y[X=0]",
#'   data.frame(X=1, Y = 1),
#'   "X-Y")
#'
process_tracing_estimator <- function(causal_model, query, data, strategies) {

  if(!requireNamespace("CausalQueries")){
    message("The process_tracing_estimator function requires the 'CausalQueries' package.")
    return(invisible())
  }

  strategy_elements <- strategies %>% lapply(function(x) strsplit(x, "-")[[1]])

  given <- CausalQueries::strategy_statements(data, strategy_elements)

  causal_model %>%
    CausalQueries::query_model(query = query, given = given) %>%
    mutate(term = strategies, XY = paste0("X", data$X, "Y", data$Y)) %>%
    select(term, XY, estimate = mean)

}



