% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pipeline_fixed_groups.R
\name{FixedGroupsPipeline}
\alias{FixedGroupsPipeline}
\title{Chain multiple transformations with different argument values per group}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Build a pipeline of transformations to be applied sequentially.

Specify different argument values for each group in a fixed set of groups.
E.g. if your \code{data.frame} contains 5 groups, you provide 5 argument values
for each of the non-constant arguments (see \code{`var_args`}).

The number of expected groups is specified during initialization and the input
\code{`data`} must be grouped such that it contains that exact number of groups.

Transformations are applied to groups separately, why the given transformation function
only receives the subset of \code{`data`} belonging to the current group.

\strong{Standard workflow}: Instantiate pipeline -> Add transformations -> Apply to data

To apply the same arguments to all groups, see
\code{\link[rearrr:Pipeline]{Pipeline}}.

To apply generated argument values to an arbitrary number of groups,
see \code{\link[rearrr:GeneratedPipeline]{GeneratedPipeline}}.
}
\examples{
# Attach package
library(rearrr)
library(dplyr)

# Create a data frame
# We group it by G so we have 3 groups
df <- data.frame(
  "Index" = 1:12,
  "A" = c(1:4, 9:12, 15:18),
  "G" = rep(1:3, each = 4)
) \%>\%
  dplyr::group_by(G)

# Create new pipeline
pipe <- FixedGroupsPipeline$new(num_groups = 3)

# Add 2D rotation transformation
pipe$add_transformation(
  fn = rotate_2d,
  args = list(
    x_col = "Index",
    y_col = "A",
    suffix = "",
    overwrite = TRUE
  ),
  var_args = list(
    degrees = list(45, 90, 180),
    origin = list(c(0, 0), c(1, 2), c(-1, 0))
  ),
  name = "Rotate"
)

# Add the `cluster_group` transformation
# As the function is fed an ungrouped subset of `data`,
# i.e. the rows of that group, we need to specify `group_cols` in `args`
# That is specific to `cluster_groups()` though
# Also note `.apply` in `var_args` which tells the pipeline *not*
# to apply this transformation to the second group
pipe$add_transformation(
  fn = cluster_groups,
  args = list(
    cols = c("Index", "A"),
    suffix = "",
    overwrite = TRUE,
    group_cols = "G"
  ),
  var_args = list(
    multiplier = list(0.5, 1, 5),
    .apply = list(TRUE, FALSE, TRUE)
  ),
  name = "Cluster"
)

# Check pipeline object
pipe

# Apply pipeline to already grouped data.frame
# Enable `verbose` to print progress
pipe$apply(df, verbose = TRUE)

}
\seealso{
Other pipelines: 
\code{\link{GeneratedPipeline}},
\code{\link{Pipeline}}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{pipelines}
\section{Super class}{
\code{\link[rearrr:Pipeline]{rearrr::Pipeline}} -> \code{FixedGroupsPipeline}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{transformations}}{\code{list} of transformations to apply.}

\item{\code{names}}{Names of the transformations.}

\item{\code{num_groups}}{Number of groups the pipeline will be applied to.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-FixedGroupsPipeline-new}{\code{FixedGroupsPipeline$new()}}
\item \href{#method-FixedGroupsPipeline-add_transformation}{\code{FixedGroupsPipeline$add_transformation()}}
\item \href{#method-FixedGroupsPipeline-apply}{\code{FixedGroupsPipeline$apply()}}
\item \href{#method-FixedGroupsPipeline-print}{\code{FixedGroupsPipeline$print()}}
\item \href{#method-FixedGroupsPipeline-clone}{\code{FixedGroupsPipeline$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-FixedGroupsPipeline-new"></a>}}
\if{latex}{\out{\hypertarget{method-FixedGroupsPipeline-new}{}}}
\subsection{Method \code{new()}}{
Initialize the pipeline with the number of groups the
pipeline will be applied to.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{FixedGroupsPipeline$new(num_groups)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{num_groups}}{Number of groups the pipeline will be applied to.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-FixedGroupsPipeline-add_transformation"></a>}}
\if{latex}{\out{\hypertarget{method-FixedGroupsPipeline-add_transformation}{}}}
\subsection{Method \code{add_transformation()}}{
Add a transformation to the pipeline.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{FixedGroupsPipeline$add_transformation(fn, args, var_args, name)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{fn}}{Function that performs the transformation.}

\item{\code{args}}{Named \code{list} with arguments for the \code{`fn`} function.}

\item{\code{var_args}}{Named \code{list} of arguments with \code{list} of differing
values for each group.

E.g. \code{list("a" = list(1, 2, 3), "b" = list("a", "b", "c"))} given 3 groups.

By adding \code{".apply"} with a list of \code{TRUE}/\code{FALSE} flags, the transformation
can be disabled for a specific group.

E.g. \code{list(".apply" = list(TRUE, FALSE, TRUE), ...}.}

\item{\code{name}}{Name of the transformation step. Must be unique.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The pipeline. To allow chaining of methods.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-FixedGroupsPipeline-apply"></a>}}
\if{latex}{\out{\hypertarget{method-FixedGroupsPipeline-apply}{}}}
\subsection{Method \code{apply()}}{
Apply the pipeline to a \code{data.frame}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{FixedGroupsPipeline$apply(data, verbose = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{data}}{\code{data.frame} with the same number of groups as pre-registered
in the pipeline.

You can find the number of groups in \code{`data`} with \code{`dplyr::n_groups(data)`}.
The number of groups expected by the pipeline can be accessed with \code{`pipe$num_groups`}.}

\item{\code{verbose}}{Whether to print the progress.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Transformed version of \code{`data`}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-FixedGroupsPipeline-print"></a>}}
\if{latex}{\out{\hypertarget{method-FixedGroupsPipeline-print}{}}}
\subsection{Method \code{print()}}{
Print an overview of the pipeline.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{FixedGroupsPipeline$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{further arguments passed to or from other methods.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The pipeline. To allow chaining of methods.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-FixedGroupsPipeline-clone"></a>}}
\if{latex}{\out{\hypertarget{method-FixedGroupsPipeline-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{FixedGroupsPipeline$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
