library(rearrr)
context("triplet_extremes()")


test_that("triplet_extremes() works", {

  # Create data frame
  xpectr::set_test_seed(1)
  df <- data.frame(
    "participant" = factor(c(1, 3, 5, 6, 7, 8, 9)),
    "score" = c(79, 85, 140, 69, 87, 92, 87),
    stringsAsFactors = FALSE
  )
  df <- df %>% dplyr::arrange(score)
  df["index"] <- 1:7

  # Single excessive element
  df_rearranged <- triplet_extremes(
    df,
    unequal_method_1 = "min"
  )
  expect_equal(df_rearranged$.triplet, factor(c(1L, 2L, 2L, 2L, 3L, 3L, 3L)))
  expect_equal(df_rearranged$index, c(1, 2, 4, 7, 3, 5, 6))
  expect_equal(df_rearranged$participant, factor(c(6, 1, 7, 5, 3, 9, 8)))
  expect_equal(df_rearranged$score, c(69, 79, 87, 140, 85, 87, 92))

  df_rearranged <- triplet_extremes(
    df,
    unequal_method_1 = "max"
  )
  expect_equal(df_rearranged$.triplet, factor(c(1, 2, 2, 2, 3, 3, 3)))
  expect_equal(df_rearranged$index, c(7, 1, 3, 6, 2, 4, 5))

  df_rearranged <- triplet_extremes(
    df,
    unequal_method_1 = "middle"
  )
  expect_equal(df_rearranged$.triplet, factor(c(1, 2, 2, 2, 3, 3, 3)))
  expect_equal(df_rearranged$index, c(4L, 1L, 3L, 7L, 2L, 5L, 6L))

  # Without excessive elements

  df <- df %>%
    dplyr::filter(dplyr::row_number() != 5) %>%
    droplevels()

  df_rearranged_min <- triplet_extremes(
    df,
    unequal_method_1 = "min"
  )
  df_rearranged_max <- triplet_extremes(
    df,
    unequal_method_1 = "max"
  )
  df_rearranged_mid <- triplet_extremes(
    df,
    unequal_method_1 = "middle"
  )
  expect_equal(df_rearranged_min, df_rearranged_max)
  expect_equal(df_rearranged_min, df_rearranged_mid)

  expect_equal(df_rearranged_min$.triplet, factor(c(1, 1, 1, 2, 2, 2)))
  expect_equal(df_rearranged_min$index, c(1L, 3L, 7L, 2L, 4L, 6L))
  expect_equal(df_rearranged_min$participant, factor(c(6, 3, 5, 1, 7, 8)))
  expect_equal(df_rearranged_min$score, c(69, 85, 140, 79, 87, 92))

})

test_that("triplet_extremes_vec() works", {
  v <- 1:20

  v_sorted <- triplet_extremes_vec(v, unequal_method_2 = c("max", "min"))

  expect_equal(v_sorted,
               triplet_extremes_vec(v, unequal_method_2 = c("min", "max")))

  ## Testing 'v_sorted'                                                     ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(v_sorted),
    "integer",
    fixed = TRUE)
  # Testing type
  expect_type(
    v_sorted,
    type = "integer")
  # Testing values
  expect_equal(
    v_sorted,
    c(1, 20, 2, 10, 19, 3, 11, 18, 4, 9, 17, 5, 12, 16, 6, 8, 15, 7,
      13, 14),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(v_sorted),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(v_sorted),
    20L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(v_sorted)),
    20L)
  ## Finished testing 'v_sorted'                                            ####

  v_sorted <- triplet_extremes_vec(v, unequal_method_2 = c("middle", "min"))

  ## Testing 'v_sorted'                                                     ####
  ## Initially generated by xpectr
  xpectr::set_test_seed(42)
  # Testing class
  expect_equal(
    class(v_sorted),
    "integer",
    fixed = TRUE)
  # Testing type
  expect_type(
    v_sorted,
    type = "integer")
  # Testing values
  expect_equal(
    v_sorted,
    c(1, 10, 2, 11, 20, 3, 12, 19, 4, 13, 18, 5, 9, 17, 6, 14, 16, 7,
      8, 15),
    tolerance = 1e-4)
  # Testing names
  expect_equal(
    names(v_sorted),
    NULL,
    fixed = TRUE)
  # Testing length
  expect_equal(
    length(v_sorted),
    20L)
  # Testing sum of element lengths
  expect_equal(
    sum(xpectr::element_lengths(v_sorted)),
    20L)
  ## Finished testing 'v_sorted'                                            ####

})

test_that("fuzz testing triplet_extremes()", {
  xpectr::set_test_seed(42)

  num_rows <- 14

  # Create a data frame
  df <- data.frame(
    "index" = 1:num_rows,
    "A" = sample(1:num_rows),
    "B" = runif(num_rows),
    "C" = LETTERS[1:num_rows],
    "G" = head(rep(c(1,2,3), each = ceiling(num_rows/3)), num_rows),
    stringsAsFactors = FALSE
  )

  dfs <- df[, c("index", "G", "A")]

  # Generate expectations for 'triplet_extremes'
  # Tip: comment out the gxs_function() call
  # so it is easy to regenerate the tests
  # xpectr::set_test_seed(42)
  # xpectr::gxs_function(
  #   fn = triplet_extremes,
  #   args_values = list(
  #     "data" = list(dfs, head(dfs, 13), head(dfs, 12), dfs$index, dplyr::group_by(dfs, G)),
  #     "col" = list("A", "index", "NOPE", NA),
  #     "middle_is" = list("middle", "min", "max", "hjsdf", NA),
  #     "unequal_method_1" = list("middle", "hjsdf", min, NA),
  #     "unequal_method_2" = list(c("middle", "middle"), c("min", "max"), c("min", "middle"), "max", c("fdsf", "fdsf"), c(min, min), c(NA, "min")),
  #     "num_groupings" = list(1, 2),
  #     "balance" = list("mean", 2, NA),
  #     "order_by_aggregates" = list(FALSE, TRUE, NA),
  #     "shuffle_members" = list(FALSE, TRUE, NA),
  #     "shuffle_triplets" = list(FALSE, TRUE, NA),
  #     "factor_name" = list(".trip", 2, NA),
  #     "overwrite" = list(FALSE)
  #   ), extra_combinations = list(
  #     list("data" = head(df, 13), "col"="C"),
  #     list("data" = df, "col"="B"),
  #     list("data" = head(dfs, 13), "unequal_method_1" = "min"),
  #     list("data" = head(dfs, 13), "unequal_method_1" = "max"),
  #     list("balance" = "spread", "num_groupings" = 2),
  #     list("balance" = "spread", "num_groupings" = 2, "order_by_aggregates" = TRUE),
  #     list("balance" = "min", "num_groupings" = 2),
  #     list("balance" = "min", "num_groupings" = 2, "order_by_aggregates" = TRUE),
  #     list("balance" = "max", "num_groupings" = 2),
  #     list("shuffle_members" = TRUE, "shuffle_triplets" = TRUE)
  #   ),
  #   indentation = 2,
  #   copy_env = FALSE
  # )




  ## Testing 'triplet_extremes'                                               ####
  ## Initially generated by xpectr
  # Testing different combinations of argument values

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19148),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19148[["index"]],
    c(5, 10, 8, 7, 2, 11, 13, 1, 12, 6, 9, 3, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["G"]],
    c(1, 2, 2, 2, 1, 3, 3, 1, 3, 2, 2, 1, 1, 3),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["A"]],
    c(7, 8, 1, 6, 14, 2, 9, 13, 3, 5, 12, 4, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_19148[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19148),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19148),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19148),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19148),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19148)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = head(dfs, 13), col =...
  # Changed from baseline: data = head(dfs, 13)
  xpectr::set_test_seed(42)
  # Assigning output
  output_19370 <- triplet_extremes(data = head(dfs, 13), col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19370),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19370[["index"]],
    c(5, 8, 7, 2, 11, 10, 1, 12, 6, 9, 3, 13, 4),
    tolerance = 1e-4)
  expect_equal(
    output_19370[["G"]],
    c(1, 2, 2, 1, 3, 2, 1, 3, 2, 2, 1, 3, 1),
    tolerance = 1e-4)
  expect_equal(
    output_19370[["A"]],
    c(7, 1, 6, 14, 2, 8, 13, 3, 5, 12, 4, 9, 10),
    tolerance = 1e-4)
  expect_equal(
    output_19370[[".trip"]],
    structure(c(1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 5L),
      .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19370),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19370),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19370),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19370),
    c(13L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19370)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = head(dfs, 12), col =...
  # Changed from baseline: data = head(dfs, 12)
  xpectr::set_test_seed(42)
  # Assigning output
  output_12861 <- triplet_extremes(data = head(dfs, 12), col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_12861),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_12861[["index"]],
    c(8, 7, 2, 11, 5, 1, 12, 6, 9, 3, 10, 4),
    tolerance = 1e-4)
  expect_equal(
    output_12861[["G"]],
    c(2, 2, 1, 3, 1, 1, 3, 2, 2, 1, 2, 1),
    tolerance = 1e-4)
  expect_equal(
    output_12861[["A"]],
    c(1, 6, 14, 2, 7, 13, 3, 5, 12, 4, 8, 10),
    tolerance = 1e-4)
  expect_equal(
    output_12861[[".trip"]],
    structure(c(1L, 1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L), .Label = c("1",
      "2", "3", "4"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_12861),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_12861),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_12861),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_12861),
    c(12L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_12861)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs$index, col = "A"...
  # Changed from baseline: data = dfs$index
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18304 <- xpectr::capture_side_effects(triplet_extremes(data = dfs$index, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18304[['error']]),
    xpectr::strip("1 assertions failed:\n * when 'data' is not a data.frame, 'col(s)' must be 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18304[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dplyr::group_by(dfs,...
  # Changed from baseline: data = dplyr::group_b...
  xpectr::set_test_seed(42)
  # Assigning output
  output_16417 <- triplet_extremes(data = dplyr::group_by(dfs, G), col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_16417),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_16417[["index"]],
    c(5, 4, 3, 1, 2, 6, 7, 8, 10, 9, 12, 11, 13, 14),
    tolerance = 1e-4)
  expect_equal(
    output_16417[["G"]],
    c(1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 3, 3, 3, 3),
    tolerance = 1e-4)
  expect_equal(
    output_16417[["A"]],
    c(7, 10, 4, 13, 14, 5, 6, 1, 8, 12, 3, 2, 9, 11),
    tolerance = 1e-4)
  expect_equal(
    output_16417[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 1L, 1L, 2L, 2L, 2L, 1L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16417),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16417),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16417),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16417),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16417)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = NULL, col = "A", mid...
  # Changed from baseline: data = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15190 <- xpectr::capture_side_effects(triplet_extremes(data = NULL, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15190[['error']]),
    xpectr::strip("Assertion failed. One of the following must apply:\n * checkmate::check_data_frame(data): Must be of type 'data.frame', not 'NULL'\n * checkmate::check_vector(data): Must be of type 'vector', not 'NULL'\n * checkmate::check_factor(data): Must be of type 'factor', not 'NULL'"),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15190[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = head(df, 13), col = ...
  # Changed from baseline: data, col
  xpectr::set_test_seed(42)
  # Assigning output
  output_17365 <- triplet_extremes(data = head(df, 13), col = "C", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_17365),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_17365[["index"]],
    c(7, 1, 6, 13, 2, 8, 12, 3, 5, 11, 4, 9, 10),
    tolerance = 1e-4)
  expect_equal(
    output_17365[["A"]],
    c(6, 13, 5, 9, 14, 1, 3, 4, 7, 2, 10, 12, 8),
    tolerance = 1e-4)
  expect_equal(
    output_17365[["B"]],
    c(0.90403, 0.46229, 0.56033, 0.3902, 0.94001, 0.13871, 0.51421,
      0.97823, 0.475, 0.08244, 0.11749, 0.98889, 0.94667),
    tolerance = 1e-4)
  # NOTE: Index for the letters:
  # 1   2   3   4   5   6   7   8   9  10  11  12  13
  # "A" "B" "C" "D" "E" "F" "G" "H" "I" "J" "K" "L" "M"
  expect_equal(
    output_17365[["C"]],
    c("G", "A", "F", "M", "B", "H", "L", "C", "E", "K", "D", "I", "J"),
    fixed = TRUE)
  expect_equal(
    output_17365[["G"]],
    c(2, 1, 2, 3, 1, 2, 3, 1, 1, 3, 1, 2, 2),
    tolerance = 1e-4)
  expect_equal(
    output_17365[[".trip"]],
    structure(c(1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 5L),
      .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_17365),
    c("index", "A", "B", "C", "G", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_17365),
    c("integer", "integer", "numeric", "character", "numeric", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_17365),
    c("integer", "integer", "double", "character", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_17365),
    c(13L, 6L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_17365)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = df, col = "B", middl...
  # Changed from baseline: data, col
  xpectr::set_test_seed(42)
  # Assigning output
  output_11346 <- triplet_extremes(data = df, col = "B", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_11346),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_11346[["index"]],
    c(12, 6, 11, 5, 9, 4, 7, 3, 8, 1, 10, 13, 14, 2),
    tolerance = 1e-4)
  expect_equal(
    output_11346[["A"]],
    c(3, 5, 2, 7, 12, 10, 6, 4, 1, 13, 8, 9, 11, 14),
    tolerance = 1e-4)
  expect_equal(
    output_11346[["B"]],
    c(0.51421, 0.56033, 0.08244, 0.475, 0.98889, 0.11749, 0.90403, 0.97823,
      0.13871, 0.46229, 0.94667, 0.3902, 0.90574, 0.94001),
    tolerance = 1e-4)
  expect_equal(
    output_11346[["C"]],
    c("L", "F", "K", "E", "I", "D", "G", "C", "H", "A", "J", "M", "N",
      "B"),
    fixed = TRUE)
  expect_equal(
    output_11346[["G"]],
    c(3, 2, 3, 1, 2, 1, 2, 1, 2, 1, 2, 3, 3, 1),
    tolerance = 1e-4)
  expect_equal(
    output_11346[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11346),
    c("index", "A", "B", "C", "G", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11346),
    c("integer", "integer", "numeric", "character", "numeric", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11346),
    c("integer", "integer", "double", "character", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11346),
    c(14L, 6L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11346)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = head(dfs, 13), col =...
  # Changed from baseline: data, unequal_method_1
  xpectr::set_test_seed(42)
  # Assigning output
  output_16569 <- triplet_extremes(data = head(dfs, 13), col = "A", middle_is = "middle", unequal_method_1 = "min", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_16569),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_16569[["index"]],
    c(8, 11, 5, 2, 12, 10, 1, 3, 7, 9, 6, 13, 4),
    tolerance = 1e-4)
  expect_equal(
    output_16569[["G"]],
    c(2, 3, 1, 1, 3, 2, 1, 1, 2, 2, 2, 3, 1),
    tolerance = 1e-4)
  expect_equal(
    output_16569[["A"]],
    c(1, 2, 7, 14, 3, 8, 13, 4, 6, 12, 5, 9, 10),
    tolerance = 1e-4)
  expect_equal(
    output_16569[[".trip"]],
    structure(c(1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 5L),
      .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16569),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16569),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16569),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16569),
    c(13L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16569)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = head(dfs, 13), col =...
  # Changed from baseline: data, unequal_method_1
  xpectr::set_test_seed(42)
  # Assigning output
  output_17050 <- triplet_extremes(data = head(dfs, 13), col = "A", middle_is = "middle", unequal_method_1 = "max", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_17050),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_17050[["index"]],
    c(2, 8, 7, 1, 11, 5, 9, 12, 6, 4, 3, 10, 13),
    tolerance = 1e-4)
  expect_equal(
    output_17050[["G"]],
    c(1, 2, 2, 1, 3, 1, 2, 3, 2, 1, 1, 2, 3),
    tolerance = 1e-4)
  expect_equal(
    output_17050[["A"]],
    c(14, 1, 6, 13, 2, 7, 12, 3, 5, 10, 4, 8, 9),
    tolerance = 1e-4)
  expect_equal(
    output_17050[[".trip"]],
    structure(c(1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 5L),
      .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_17050),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_17050),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_17050),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_17050),
    c(13L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_17050)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "index", ...
  # Changed from baseline: col = "index"
  xpectr::set_test_seed(42)
  # Assigning output
  output_14577 <- triplet_extremes(data = dfs, col = "index", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_14577),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_14577[["index"]],
    c(7, 8, 1, 6, 14, 2, 9, 13, 3, 5, 12, 4, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_14577[["G"]],
    c(2, 2, 1, 2, 3, 1, 2, 3, 1, 1, 3, 1, 2, 3),
    tolerance = 1e-4)
  expect_equal(
    output_14577[["A"]],
    c(6, 1, 13, 5, 11, 14, 12, 9, 4, 7, 3, 10, 8, 2),
    tolerance = 1e-4)
  expect_equal(
    output_14577[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_14577),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_14577),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_14577),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_14577),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_14577)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "NOPE", m...
  # Changed from baseline: col = "NOPE"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17191 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "NOPE", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17191[['error']]),
    xpectr::strip("1 assertions failed:\n * These names in the 'col(s)' argument were not found in 'data': NOPE."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17191[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = NA, middl...
  # Changed from baseline: col = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19346 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = NA, middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19346[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'cols': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19346[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = NULL, mid...
  # Changed from baseline: col = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_12554 <- triplet_extremes(data = dfs, col = NULL, middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_12554),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_12554[["index"]],
    c(7, 8, 1, 6, 14, 2, 9, 13, 3, 5, 12, 4, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_12554[["G"]],
    c(2, 2, 1, 2, 3, 1, 2, 3, 1, 1, 3, 1, 2, 3),
    tolerance = 1e-4)
  expect_equal(
    output_12554[["A"]],
    c(6, 1, 13, 5, 11, 14, 12, 9, 4, 7, 3, 10, 8, 2),
    tolerance = 1e-4)
  expect_equal(
    output_12554[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_12554),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_12554),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_12554),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_12554),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_12554)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: middle_is = "hjsdf"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14622 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "hjsdf", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_14622[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(is_checkmate_v2_1(),
             "must be a subset of {'min','middle','max'}, but has additional elements {'hjsdf'}.",
             "must be a subset of set {min,middle,max}"), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14622[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: middle_is = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19400 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = NA, unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19400[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'middle_is': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19400[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: middle_is = "min"
  xpectr::set_test_seed(42)
  # Assigning output
  output_19782 <- triplet_extremes(data = dfs, col = "A", middle_is = "min", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19782),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19782[["index"]],
    c(5, 10, 8, 11, 2, 12, 3, 1, 6, 7, 9, 13, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_19782[["G"]],
    c(1, 2, 2, 3, 1, 3, 1, 1, 2, 2, 2, 3, 1, 3),
    tolerance = 1e-4)
  expect_equal(
    output_19782[["A"]],
    c(7, 8, 1, 2, 14, 3, 4, 13, 5, 6, 12, 9, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_19782[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19782),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19782),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19782),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19782),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19782)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: middle_is = "max"
  xpectr::set_test_seed(42)
  # Assigning output
  output_11174 <- triplet_extremes(data = dfs, col = "A", middle_is = "max", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_11174),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_11174[["index"]],
    c(5, 10, 8, 1, 2, 11, 14, 9, 12, 13, 4, 3, 6, 7),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["G"]],
    c(1, 2, 2, 1, 1, 3, 3, 2, 3, 3, 1, 1, 2, 2),
    tolerance = 1e-4)
  expect_equal(
    output_11174[["A"]],
    c(7, 8, 1, 13, 14, 2, 11, 12, 3, 9, 10, 4, 5, 6),
    tolerance = 1e-4)
  expect_equal(
    output_11174[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_11174),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_11174),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_11174),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_11174),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_11174)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: middle_is = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14749 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = NULL, unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'middle_is': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14749[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_1 = "h...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15603 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "hjsdf", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_15603[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'min','middle','max'}, but has additional elements {'hjsdf'}.",
        "must be a subset of set {min,middle,max}."
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15603[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_1 = min
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19040 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = min, unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19040[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_1': Must be of type 'string', not 'builtin'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19040[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_1 = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_11387 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = NA, unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11387[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_1': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_11387[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_1 = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19888 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = NULL, unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19888[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_1': Must be of type 'string', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19888[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = c(...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19466 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("min", "max"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19466),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19466[["index"]],
    c(8, 2, 11, 5, 1, 12, 10, 9, 3, 7, 14, 6, 13, 4),
    tolerance = 1e-4)
  expect_equal(
    output_19466[["G"]],
    c(2, 1, 3, 1, 1, 3, 2, 2, 1, 2, 3, 2, 3, 1),
    tolerance = 1e-4)
  expect_equal(
    output_19466[["A"]],
    c(1, 14, 2, 7, 13, 3, 8, 12, 4, 6, 11, 5, 9, 10),
    tolerance = 1e-4)
  expect_equal(
    output_19466[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19466),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19466),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19466),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19466),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19466)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = c(...
  xpectr::set_test_seed(42)
  # Assigning output
  output_10824 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("min", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_10824),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_10824[["index"]],
    c(8, 10, 11, 5, 2, 12, 13, 1, 3, 7, 9, 6, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_10824[["G"]],
    c(2, 2, 3, 1, 1, 3, 3, 1, 1, 2, 2, 2, 1, 3),
    tolerance = 1e-4)
  expect_equal(
    output_10824[["A"]],
    c(1, 8, 2, 7, 14, 3, 9, 13, 4, 6, 12, 5, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_10824[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_10824),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_10824),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_10824),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_10824),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_10824)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = "max"
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_15142 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = "max", num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15142[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_2': Must have length 2, but has length 1."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_15142[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = c(...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13902 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("fdsf", "fdsf"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_match(
    xpectr::strip(side_effects_13902[['error']], lowercase = TRUE),
    xpectr::strip(
      ifelse(
        is_checkmate_v2_1(),
        "must be a subset of {'min','middle','max'}, but has additional elements {'fdsf'}.",
        "must be a subset of set {min,middle,max}."
      ), lowercase = TRUE),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13902[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = c(...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19057 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c(min, min), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19057[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_2': Must be of type 'character', not 'list'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19057[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = c(...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14469 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c(NA, "min"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14469[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_2': Must have names, but is NA at position 1."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14469[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: unequal_method_2 = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18360 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = NULL, num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18360[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'unequal_method_2': Must be of type 'character', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18360[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Initial groupings are ordered by group identifiers
  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings = 2
  xpectr::set_test_seed(42)
  # Assigning output
  output_17375 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 2, balance = "mean", shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_17375),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_17375[["index"]],
    c(8, 7, 2, 12, 6, 9, 5, 10, 11, 13, 1, 3, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_17375[["G"]],
    c(2, 2, 1, 3, 2, 2, 1, 2, 3, 3, 1, 1, 1, 3),
    tolerance = 1e-4)
  # 1, 6, 14 = 21 ; 3, 5, 12 = 20 ; 7, 8 = 15 ; 2, 9, 13 = 24 ; 4, 10, 11 = 25
  expect_equal(
    output_17375[["A"]],
    c(1, 6, 14, 3, 5, 12, 7, 8, 2, 9, 13, 4, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_17375[[".trip_1"]],
    structure(c(2L, 2L, 2L, 4L, 4L, 4L, 1L, 1L, 3L, 3L, 3L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_17375[[".trip_2"]],
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_17375),
    c("index", "G", "A", ".trip_1", ".trip_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_17375),
    c("integer", "numeric", "integer", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_17375),
    c("integer", "double", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_17375),
    c(14L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_17375)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_18110 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = NULL, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18110[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'num_groupings': Must be of type 'count', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_18110[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings, balance
  xpectr::set_test_seed(42)
  # Assigning output
  output_13881 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 2, balance = "spread", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_13881),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_13881[["index"]],
    c(12, 6, 9, 3, 4, 14, 5, 10, 8, 7, 2, 11, 13, 1),
    tolerance = 1e-4)
  expect_equal(
    output_13881[["G"]],
    c(3, 2, 2, 1, 1, 3, 1, 2, 2, 2, 1, 3, 3, 1),
    tolerance = 1e-4)
  # 12-3 = 9 ; 11-4 = 7 ; 8-7 = 1 ; 14-1 = 13 ; 13-2 = 11;
  expect_equal(
    output_13881[["A"]],
    c(3, 5, 12, 4, 10, 11, 7, 8, 1, 6, 14, 2, 9, 13),
    tolerance = 1e-4)
  expect_equal(
    output_13881[[".trip_1"]],
    structure(c(4L, 4L, 4L, 5L, 5L, 5L, 1L, 1L, 2L, 2L, 2L, 3L, 3L,
      3L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_13881[[".trip_2"]],
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_13881),
    c("index", "G", "A", ".trip_1", ".trip_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_13881),
    c("integer", "numeric", "integer", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_13881),
    c("integer", "double", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_13881),
    c(14L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_13881)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings, balance
  xpectr::set_test_seed(42)
  # Assigning output
  output_16851 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 2, balance = "min", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_16851),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_16851[["index"]],
    c(11, 13, 1, 12, 6, 9, 5, 10, 8, 7, 2, 3, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_16851[["G"]],
    c(3, 3, 1, 3, 2, 2, 1, 2, 2, 2, 1, 1, 1, 3),
    tolerance = 1e-4)
  # 2, 3, 7, 1, 4
  expect_equal(
    output_16851[["A"]],
    c(2, 9, 13, 3, 5, 12, 7, 8, 1, 6, 14, 4, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_16851[[".trip_1"]],
    structure(c(3L, 3L, 3L, 4L, 4L, 4L, 1L, 1L, 2L, 2L, 2L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_16851[[".trip_2"]],
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16851),
    c("index", "G", "A", ".trip_1", ".trip_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16851),
    c("integer", "numeric", "integer", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16851),
    c("integer", "double", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16851),
    c(14L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16851)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings, balance
  xpectr::set_test_seed(42)
  # Assigning output
  output_10039 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 2, balance = "max", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_10039),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_10039[["index"]],
    c(12, 6, 9, 3, 4, 14, 5, 10, 8, 7, 2, 11, 13, 1),
    tolerance = 1e-4)
  expect_equal(
    output_10039[["G"]],
    c(3, 2, 2, 1, 1, 3, 1, 2, 2, 2, 1, 3, 3, 1),
    tolerance = 1e-4)
  # 12, 11, 8, 14, 13
  expect_equal(
    output_10039[["A"]],
    c(3, 5, 12, 4, 10, 11, 7, 8, 1, 6, 14, 2, 9, 13),
    tolerance = 1e-4)
  expect_equal(
    output_10039[[".trip_1"]],
    structure(c(4L, 4L, 4L, 5L, 5L, 5L, 1L, 1L, 2L, 2L, 2L, 3L, 3L,
      3L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_10039[[".trip_2"]],
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_10039),
    c("index", "G", "A", ".trip_1", ".trip_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_10039),
    c("integer", "numeric", "integer", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_10039),
    c("integer", "double", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_10039),
    c(14L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_10039)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings, balanc...
  xpectr::set_test_seed(42)
  # Assigning output
  output_18329 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 2, balance = "spread", order_by_aggregates = TRUE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_18329),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_18329[["index"]],
    c(3, 4, 14, 12, 6, 9, 5, 10, 11, 13, 1, 8, 7, 2),
    tolerance = 1e-4)
  expect_equal(
    output_18329[["G"]],
    c(1, 1, 3, 3, 2, 2, 1, 2, 3, 3, 1, 2, 2, 1),
    tolerance = 1e-4)
  # 11-4=7 ; 12-3=9 ; 8-7=1 ; 13-2=11 ; 14-1=13
  expect_equal(
    output_18329[["A"]],
    c(4, 10, 11, 3, 5, 12, 7, 8, 2, 9, 13, 1, 6, 14),
    tolerance = 1e-4)
  expect_equal(
    output_18329[[".trip_1"]],
    structure(c(5L, 5L, 5L, 4L, 4L, 4L, 1L, 1L, 3L, 3L, 3L, 2L, 2L,
      2L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_18329[[".trip_2"]],
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_18329),
    c("index", "G", "A", ".trip_1", ".trip_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_18329),
    c("integer", "numeric", "integer", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_18329),
    c("integer", "double", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_18329),
    c(14L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_18329)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: num_groupings, balanc...
  xpectr::set_test_seed(42)
  # Assigning output
  output_10073 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 2, balance = "min", order_by_aggregates = TRUE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_10073),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_10073[["index"]],
    c(11, 13, 1, 12, 6, 9, 8, 7, 2, 3, 4, 14, 5, 10),
    tolerance = 1e-4)
  expect_equal(
    output_10073[["G"]],
    c(3, 3, 1, 3, 2, 2, 2, 2, 1, 1, 1, 3, 1, 2),
    tolerance = 1e-4)
  # 2 ; 3 ; 1 ; 4 ; 7
  expect_equal(
    output_10073[["A"]],
    c(2, 9, 13, 3, 5, 12, 1, 6, 14, 4, 10, 11, 7, 8),
    tolerance = 1e-4)
  expect_equal(
    output_10073[[".trip_1"]],
    structure(c(3L, 3L, 3L, 4L, 4L, 4L, 2L, 2L, 2L, 5L, 5L, 5L, 1L,
      1L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  expect_equal(
    output_10073[[".trip_2"]],
    structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L,
      2L), .Label = c("1", "2"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_10073),
    c("index", "G", "A", ".trip_1", ".trip_2"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_10073),
    c("integer", "numeric", "integer", "factor", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_10073),
    c("integer", "double", "integer", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_10073),
    c(14L, 5L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_10073)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: balance = 2
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_12076 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = 2, order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12076[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'balance': Must be of type 'character', not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_12076[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: balance = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19066 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = NA, order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19066[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'balance': Contains missing values (element 1)."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19066[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: balance = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16117 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = NULL, order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16117[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'balance': Must be of type 'character', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16117[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: order_by_aggregates =...
  xpectr::set_test_seed(42)
  # Assigning output
  output_19148 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = TRUE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19148),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19148[["index"]],
    c(5, 10, 8, 7, 2, 11, 13, 1, 12, 6, 9, 3, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["G"]],
    c(1, 2, 2, 2, 1, 3, 3, 1, 3, 2, 2, 1, 1, 3),
    tolerance = 1e-4)
  expect_equal(
    output_19148[["A"]],
    c(7, 8, 1, 6, 14, 2, 9, 13, 3, 5, 12, 4, 10, 11),
    tolerance = 1e-4)
  expect_equal(
    output_19148[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19148),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19148),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19148),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19148),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19148)),
    character(0),
    fixed = TRUE)
  ## Finished testing 'triplet_extremes(data = dfs, col = "A", midd...'     ####

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: order_by_aggregates = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14357 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = NA, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14357[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'order_by_aggregates': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14357[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: order_by_aggregates =...
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_10374 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = NULL, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10374[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'order_by_aggregates': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_10374[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_members = TRUE
  xpectr::set_test_seed(42)
  # Assigning output
  output_19735 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = TRUE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_19735),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_19735[["index"]],
    c(10, 5, 8, 7, 2, 13, 1, 11, 6, 9, 12, 14, 4, 3),
    tolerance = 1e-4)
  expect_equal(
    output_19735[["G"]],
    c(2, 1, 2, 2, 1, 3, 1, 3, 2, 2, 3, 3, 1, 1),
    tolerance = 1e-4)
  expect_equal(
    output_19735[["A"]],
    c(8, 7, 1, 6, 14, 9, 13, 2, 5, 12, 3, 11, 10, 4),
    tolerance = 1e-4)
  expect_equal(
    output_19735[[".trip"]],
    structure(c(1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L,
      5L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_19735),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_19735),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_19735),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_19735),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_19735)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_members = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_14317 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = NA, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14317[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'shuffle_members': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_14317[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_members = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19575 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = NULL, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19575[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'shuffle_members': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19575[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_members, shuf...
  xpectr::set_test_seed(42)
  # Assigning output
  output_18877 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = TRUE, shuffle_triplets = TRUE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_18877),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_18877[["index"]],
    c(13, 1, 11, 6, 9, 12, 14, 4, 3, 10, 5, 8, 7, 2),
    tolerance = 1e-4)
  expect_equal(
    output_18877[["G"]],
    c(3, 1, 3, 2, 2, 3, 3, 1, 1, 2, 1, 2, 2, 1),
    tolerance = 1e-4)
  expect_equal(
    output_18877[["A"]],
    c(9, 13, 2, 5, 12, 3, 11, 10, 4, 8, 7, 1, 6, 14),
    tolerance = 1e-4)
  expect_equal(
    output_18877[[".trip"]],
    structure(c(3L, 3L, 3L, 4L, 4L, 4L, 5L, 5L, 5L, 1L, 1L, 2L, 2L,
      2L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_18877),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_18877),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_18877),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_18877),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_18877)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_triplets = TRUE
  xpectr::set_test_seed(42)
  # Assigning output
  output_16399 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = TRUE, factor_name = ".trip", overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_16399),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_16399[["index"]],
    c(3, 4, 14, 12, 6, 9, 5, 10, 8, 7, 2, 11, 13, 1),
    tolerance = 1e-4)
  expect_equal(
    output_16399[["G"]],
    c(1, 1, 3, 3, 2, 2, 1, 2, 2, 2, 1, 3, 3, 1),
    tolerance = 1e-4)
  expect_equal(
    output_16399[["A"]],
    c(4, 10, 11, 3, 5, 12, 7, 8, 1, 6, 14, 2, 9, 13),
    tolerance = 1e-4)
  expect_equal(
    output_16399[[".trip"]],
    structure(c(5L, 5L, 5L, 4L, 4L, 4L, 1L, 1L, 2L, 2L, 2L, 3L, 3L,
      3L), .Label = c("1", "2", "3", "4", "5"), class = "factor"))
  # Testing column names
  expect_equal(
    names(output_16399),
    c("index", "G", "A", ".trip"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_16399),
    c("integer", "numeric", "integer", "factor"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_16399),
    c("integer", "double", "integer", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_16399),
    c(14L, 4L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_16399)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_triplets = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_19709 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = NA, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19709[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'shuffle_triplets': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_19709[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: shuffle_triplets = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_16188 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = NULL, factor_name = ".trip", overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16188[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'shuffle_triplets': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_16188[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: factor_name = 2
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13334 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = 2, overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13334[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'factor_name': Must be of type 'string' (or 'NULL'), not 'double'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13334[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: factor_name = NA
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_13467 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = NA, overwrite = FALSE), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13467[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'factor_name': May not be NA."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_13467[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: factor_name = NULL
  xpectr::set_test_seed(42)
  # Assigning output
  output_13984 <- triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = NULL, overwrite = FALSE)
  # Testing class
  expect_equal(
    class(output_13984),
    c("tbl_df", "tbl", "data.frame"),
    fixed = TRUE)
  # Testing column values
  expect_equal(
    output_13984[["index"]],
    c(5, 10, 8, 7, 2, 11, 13, 1, 12, 6, 9, 3, 4, 14),
    tolerance = 1e-4)
  expect_equal(
    output_13984[["G"]],
    c(1, 2, 2, 2, 1, 3, 3, 1, 3, 2, 2, 1, 1, 3),
    tolerance = 1e-4)
  expect_equal(
    output_13984[["A"]],
    c(7, 8, 1, 6, 14, 2, 9, 13, 3, 5, 12, 4, 10, 11),
    tolerance = 1e-4)
  # Testing column names
  expect_equal(
    names(output_13984),
    c("index", "G", "A"),
    fixed = TRUE)
  # Testing column classes
  expect_equal(
    xpectr::element_classes(output_13984),
    c("integer", "numeric", "integer"),
    fixed = TRUE)
  # Testing column types
  expect_equal(
    xpectr::element_types(output_13984),
    c("integer", "double", "integer"),
    fixed = TRUE)
  # Testing dimensions
  expect_equal(
    dim(output_13984),
    c(14L, 3L))
  # Testing group keys
  expect_equal(
    colnames(dplyr::group_keys(output_13984)),
    character(0),
    fixed = TRUE)

  # Testing triplet_extremes(data = dfs, col = "A", midd...
  # Changed from baseline: overwrite = NULL
  xpectr::set_test_seed(42)
  # Testing side effects
  # Assigning side effects
  side_effects_17846 <- xpectr::capture_side_effects(triplet_extremes(data = dfs, col = "A", middle_is = "middle", unequal_method_1 = "middle", unequal_method_2 = c("middle", "middle"), num_groupings = 1, balance = "mean", order_by_aggregates = FALSE, shuffle_members = FALSE, shuffle_triplets = FALSE, factor_name = ".trip", overwrite = NULL), reset_seed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17846[['error']]),
    xpectr::strip("1 assertions failed:\n * Variable 'overwrite': Must be of type 'logical flag', not 'NULL'."),
    fixed = TRUE)
  expect_equal(
    xpectr::strip(side_effects_17846[['error_class']]),
    xpectr::strip(c("simpleError", "error", "condition")),
    fixed = TRUE)

  ## Finished testing 'triplet_extremes'                                      ####
  #

  #

})

