#' @name exportArms
#' @aliases exportArms.redcapApiConnection
#' @aliases exportArms.redcapDbConnection
#' @export exportArms
#' @export exportArms.redcapApiConnection
#' @export exportArms.redcapDbConnection
#' @importFrom httr POST
#' 
#' @title Exports the Arms for a Project
#' @description Retrieve a data frame giving the event IDs and event names
#'   of a project
#'   
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}
#' @param arms A vector of arm numbers that you wish to pull events for (by default, 
#' all events are pulled).  A bug exists in early versions of the API that causes
#' all arms to be returned regardless of this argument.  This bug was fixed in 
#' version 5.9.15
#' @param ... Arguments to be passed to other methods
#' 
#' @details The data frame that is returned shows the arm number, and arm name.
#' 
#' When this function is called for a classic project, a character string is 
#' returned giving the API error message, '400: You cannot export arms for 
#' classic projects' but without casting an error in R. This is by design and 
#' allows more flexible error checks in certain functions.
#' 
#' @author Benjamin Nutter
#' 
#' @references
#' Please refer to your institution's API documentation.
#'   
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @examples
#'   \dontrun{
#'   > #*** Note: I cannot provide working examples without
#'   > #*** compromising security.  Instead, I will try to 
#'   > #*** offer up sample code with the matching results
#'   > 
#'   > 
#'   > #*** Create the connection object
#'   > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#'   > 
#'   > exportArms(rcon)
#'   arm_num             name
#'   1       1            Arm 1
#'   2       2 Experimental Arm
#'   3      10   Normal Control
#'   }
  
exportArms <- function(rcon, arms, ...) UseMethod("exportArms")

#' @rdname exportArms

exportArms.redcapDbConnection <- function(rcon, arms, ...){
  message("Please accept my apologies.  The exportUsers method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname exportArms

exportArms.redcapApiConnection <- function(rcon, arms, ...){
  .params <- list(token=rcon$token, content='arm', format='csv', returnFormat='csv')
  if (!missing(arms)) .params[['arms']] <- paste(arms, collapse=',')
  x <- httr::POST(url=rcon$url, body=.params, config=rcon$config)
  
  if (x$status_code == "200")
    return(read.csv(textConnection(as.character(x)), stringsAsFactors=FALSE))
  #*** For classic projects, we want to avoid throwing a disruptive error. Instead, we 
  #*** return the message that indicates this is a classic project.
  else if (x$status_code == "400" & as.character(x) == "You cannot export arms for classic projects") 
    paste0(x$status_code, ": ", as.character(x))
  else stop(paste(x$status_code, ": ", as.character(x), sep=""))
}
