% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/qcPrec.R
\name{qcPrec}
\alias{qcPrec}
\title{Quality Control of daily precipitation observations}
\usage{
qcPrec(
  prec,
  sts,
  crs,
  coords,
  coords_as_preds = TRUE,
  neibs = 10,
  thres = NA,
  qc = "all",
  qc3 = 10,
  qc4 = c(0.99, 5),
  qc5 = c(0.01, 0.1, 5),
  ncpu = 1
)
}
\arguments{
\item{prec}{matrix containing the original precipitation data. Each column represents one station. The names of columns have to be names of the stations.}

\item{sts}{data.frame. A column "ID" (unique ID of stations) is required. The rest of the columns (all of them) will act as predictors of the model.}

\item{crs}{character. Coordinates system in EPSG format (e.g.: "EPSG:4326").}

\item{coords}{vector of two character elements. Names of the fields in "sts" containing longitude and latitude.}

\item{coords_as_preds}{logical. If TRUE (default), "coords" are also taken as predictors.}

\item{neibs}{integer. Number of nearest neighbors to use.}

\item{thres}{numeric. Maximum radius (in km) where neighboring stations will be searched. NA value uses the whole spatial domain.}

\item{qc}{vector of strings with the QC criteria to apply. Default is "all". See details.}

\item{qc3}{numeric. Indicates the threshold (number of times higher or lower) from which a observation, in comparison with its estimate, should be deleted. Default is 10.}

\item{qc4}{numeric vector of length 2. Thresholds of wet probability (0 to 1) and magnitude (in the units of input precipitation data) from which a observation of value zero, in comparison with its estimate, should be deleted. Default is c(0.99, 5).}

\item{qc5}{numeric vector of length 2. Thresholds of dry probability (0 to 1) and magnitude (in the units of input precipitation data) from which a observation higher than a specific value (also in the original units), in comparison with its estimate, should be deleted. Default is c(0.01, 0.1, 5).}

\item{ncpu}{number of processor cores used to parallel computing.}
}
\description{
This function apply several threshold-based criteria to filter original observations of daily precipitation.
}
\details{
Parameter "sts" must have an "ID" field containing unique identifiers of the stations.

"qc" can be "all" (all criteria are applied) or a vector of strings (e.g.: c("1","2","4")) indicating the QC criteria to apply to observations:
"1" (suspect value): obs==0 & all(neibs>0);
"2" (suspect zero): obs>0 & all(neibs==0);
"3" (suspect outlier): obs is "qc3" times higher or lower than the estimate;
"4" (suspect wet): obs==0 & wet probability > "qc4[1]" & estimate > "qc4[2]";
"5" (suspect dry): obs>"qc5[3]" & dry probability < "qc5[1]" & estimate < "qc5[2]"
}
\examples{
\dontrun{
set.seed(123)
prec <- round(matrix(rnorm(30*50, mean = 1.2, sd = 6), 30, 50), 1)
prec[prec<0] <- 0
colnames(prec) <- paste0('sts_',1:50) 
sts <- data.frame(ID = paste0('sts_',1:50), lon = rnorm(50,0,1), 
                  lat = rnorm(50,40,1), dcoast = rnorm(50,200,50))
qcdata <- qcPrec(prec, sts, crs = 'EPSG:4326', coords = c('lon','lat'),
                 coords_as_preds = TRUE, neibs = 10, thres = NA,
                 qc = 'all', qc3 = 10, qc4 = c(0.99, 5), qc5 = c(0.01, 0.1, 5),
                 ncpu=2)
str(qcdata)
}
}
