% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fpc.R
\name{fpc}
\alias{fpc}
\title{Construct a FPC regression term}
\usage{
fpc(X, argvals = NULL, method = c("svd", "fpca.sc", "fpca.face",
  "fpca.ssvd"), ncomp = NULL, pve = 0.99, penalize = (method ==
  "svd"), bs = "ps", k = 40, ...)
}
\arguments{
\item{X}{functional predictors, typically expressed as an \code{N} by \code{J} matrix,
where \code{N} is the number of columns and \code{J} is the number of
evaluation points. May include missing/sparse functions, which are
indicated by \code{NA} values. Alternatively, can be an object of class
\code{"fd"}; see \code{\link[fda]{fd}}.}

\item{argvals}{indices of evaluation of \code{X}, i.e. \eqn{(t_{i1},.,t_{iJ})} for
subject \eqn{i}. May be entered as either a length-\code{J} vector, or as
an \code{N} by \code{J} matrix. Indices may be unequally spaced. Entering
as a matrix allows for different observations times for each subject. If
\code{NULL}, defaults to an equally-spaced grid between 0 or 1 (or within
\code{X$basis$rangeval} if \code{X} is a \code{fd} object.)}

\item{method}{the method used for finding principal components. The default
is an unconstrained SVD of the \eqn{XB} matrix. Alternatives include
constrained (functional) principal components approaches}

\item{ncomp}{number of principal components. if \code{NULL}, chosen by \code{pve}}

\item{pve}{proportion of variance explained; used to choose the number of
principal components}

\item{penalize}{if \code{TRUE}, a roughness penalty is applied to the
functional estimate. Defaults to \code{TRUE} if \code{method=="svd"}
(corresponding to the FPCR_R method of Reiss and Ogden (2007)), and
\code{FALSE} if \code{method!="svd"} (corresponding to FPCR_C).}

\item{bs}{two letter character string indicating the \code{mgcv}-style basis
to use for pre-smoothing \code{X}}

\item{k}{the dimension of the pre-smoothing basis}

\item{...}{additional options to be passed to \code{\link{lf}}. These include
\code{argvals}, \code{integration}, and any additional options for the
pre-smoothing basis (as constructed by \code{mgcv::s}), such as \code{m}.}
}
\value{
The result of a call to \code{\link{lf}}.
}
\description{
Constructs a functional principal component regression (Reiss and Ogden, 
2007, 2010) term for inclusion in an \code{mgcv::gam}-formula (or
\code{\link{bam}} or \code{\link{gamm}} or \code{gamm4:::gamm}) as
constructed by \code{\link{pfr}}. Currently only one-dimensional functions
are allowed.
}
\details{
\code{fpc} is a wrapper for \code{\link{lf}}, which defines linear
functional predictors for any type of basis for inclusion in a \code{pfr}
formula. \code{fpc} simply calls \code{lf} with the appropriate options for
the \code{fpc} basis and penalty construction.

This function implements both the FPCR-R and FPCR-C methods of Reiss and
Ogden (2007). Both methods consist of the following steps:
\enumerate{
 \item project \eqn{X} onto a spline basis \eqn{B}
 \item perform a principal components decomposition of \eqn{XB}
 \item use those PC's as the basis in fitting a (generalized) functional
   linear model
}

This implementation provides options for each of these steps. The basis
for in step 1 can be specified using the arguments \code{bs} and \code{k},
as well as other options via \code{...}; see \code{\link[mgcv]{s}} for
these options. The type of PC-decomposition is specified with \code{method}.
And the FLM can be fit either penalized or unpenalized via \code{penalize}.

The default is FPCR-R, which uses a b-spline basis, an unconstrained 
principal components decomposition using \code{\link{svd}}, and the FLM
fit with a second-order difference penalty. FPCR-C can be selected by
using a different option for \code{method}, indicating a constrained
("functional") PC decomposition, and by default an unpenalized fit of the
FLM.

FPCR-R is also implemented in \code{\link{fpcr}}; here we implement the
  method for inclusion in a \code{pfr} formula.
}
\section{NOTE}{

Unlike \code{\link{fpcr}}, \code{fpc} within a \code{pfr} formula does
not automatically decorrelate the functional predictors from additional
scalar covariates.
}

\examples{
data(gasoline)
par(mfrow=c(3,1))

# Fit PFCR_R
gasmod1 <- pfr(octane ~ fpc(NIR, ncomp=30), data=gasoline)
plot(gasmod1, rug=FALSE)
est1 <- coef(gasmod1)

# Fit FPCR_C with fpca.sc
gasmod2 <- pfr(octane ~ fpc(NIR, method="fpca.sc", ncomp=6), data=gasoline)
plot(gasmod2, se=FALSE)
est2 <- coef(gasmod2)

# Fit penalized model with fpca.face
gasmod3 <- pfr(octane ~ fpc(NIR, method="fpca.face", penalize=TRUE), data=gasoline)
plot(gasmod3, rug=FALSE)
est3 <- coef(gasmod3)

par(mfrow=c(1,1))
ylm <- range(est1$value)*1.35
plot(value ~ X.argvals, type="l", data=est1, ylim=ylm)
lines(value ~ X.argvals, col=2, data=est2)
lines(value ~ X.argvals, col=3, data=est3)

}
\references{
Reiss, P. T. (2006). Regression with signals and images as predictors. Ph.D.
dissertation, Department of Biostatistics, Columbia University. Available
at http://works.bepress.com/phil_reiss/11/.

Reiss, P. T., and Ogden, R. T. (2007). Functional principal component
regression and functional partial least squares. \emph{Journal of the
American Statistical Association}, 102, 984-996.

Reiss, P. T., and Ogden, R. T. (2010). Functional generalized linear models
with images as predictors. \emph{Biometrics}, 66, 61-69.
}
\seealso{
\code{\link{lf}}, \code{\link{smooth.construct.fpc.smooth.spec}}
}
\author{
Jonathan Gellar \email{JGellar@mathematica-mpr.com}, Phil Reiss
  \email{phil.reiss@nyumc.org}, Lan Huo \email{lan.huo@nyumc.org}, and
  Lei Huang \email{huangracer@gmail.com}
}
