% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/environment_gridworld.R
\name{Gridworld}
\alias{Gridworld}
\title{Gridworld}
\arguments{
\item{shape}{[\code{integer(2)}] \cr
Shape of the gridworld (number of rows x number of columns).}

\item{goal.states}{[\code{integer}] \cr
Goal states in the gridworld.}

\item{cliff.states}{[\code{integer}] \cr
Cliff states in the gridworld.}

\item{reward.step}{[\code{integer(1)}] \cr
Reward for taking a step.}

\item{cliff.transition.states}{[\code{integer}] \cr
States to which the environment transitions if stepping into the cliff.
If it is a vector, all states will have equal probability.
Only used when \code{cliff.transition.done == FALSE},
else specify the \code{initial.state} argument.}

\item{reward.cliff}{[\code{integer(1)}] \cr
Reward for taking a step in the cliff state.}

\item{diagonal.moves}{[\code{logical(1)}] \cr
Should diagonal moves be allowed?}

\item{wind}{[\code{integer}] \cr
Strength of the upward wind in each cell.}

\item{cliff.transition.done}{[\code{logical(1)}] \cr
Should the episode end after stepping into the cliff?}

\item{stochasticity}{[\code{numeric(1)}] \cr
Probability of random transition to any of the neighboring states when taking any action.}

\item{...}{[\code{any}] \cr Arguments passed on to \link{makeEnvironment}.}
}
\description{
Creates gridworld environments.
}
\details{
A gridworld is an episodic navigation task, the goal is to get from start state to goal state.

Possible actions include going left, right, up or down. If \code{diagonal.moves = TRUE} diagonal
moves are also possible, leftup, leftdown, rightup and rightdown.

When stepping into a cliff state you get a reward of \code{reward.cliff},
usually a high negative reward and transition to a state specified by \code{cliff.transition.states}.

In each column a deterministic wind specified via \code{wind} pushes you up a specific number of
grid cells (for the next action).

A stochastic gridworld is a gridworld where with probability \code{stochasticity} the next state
is chosen at random from all neighbor states independent of the actual action.

If an action would take you off the grid, the new state is the nearest cell inside the grid.
For each step you get a reward of \code{reward.step}, until you reach a goal state,
then the episode is done.

States are enumerated row-wise and numeration starts with 0.
Here is an example 4x4 grid:
\tabular{rrrr}{
0 \tab 1 \tab 2 \tab 3 \cr
4 \tab 5 \tab 6 \tab 7 \cr
8 \tab 9 \tab 10 \tab 11 \cr
12 \tab 13 \tab 14 \tab 15 \cr
}
So a board position could look like this (G: goal state, x: current state, C: cliff state):
\tabular{rrrr}{
G \tab o \tab o \tab o \cr
o \tab o \tab o \tab o \cr
o \tab x \tab o \tab o \cr
o \tab o \tab o \tab C \cr
}
}
\section{Usage}{

\code{makeEnvironment("gridworld", shape = NULL, goal.states = NULL, cliff.states = NULL, reward.step = -1, reward.cliff = -100, diagonal.moves = FALSE, wind = rep(0, shape[2]), cliff.transition.states = NULL, cliff.transition.done = FALSE, stochasticity = 0, ...)}
}

\section{Methods}{

\itemize{
\item \code{$step(action)} \cr
Take action in environment.
Returns a list with \code{state}, \code{reward}, \code{done}.
\item \code{$reset()} \cr
Resets the \code{done} flag of the environment and returns an initial state.
Useful when starting a new episode.
\item \code{$visualize()} \cr
Visualizes the environment (if there is a visualization function).
}
}

\examples{
# Gridworld Environment (Sutton & Barto Example 4.1)
env1 = makeEnvironment("gridworld", shape = c(4L, 4L), goal.states = 0L,
  initial.state = 15L)
env1$reset()
env1$visualize()
env1$step(0L)
env1$visualize()

# Windy Gridworld (Sutton & Barto Example 6.5)
env2 = makeEnvironment("gridworld", shape = c(7, 10), goal.states = 37L,
  reward.step = -1, wind = c(0, 0, 0, 1, 1, 1, 2, 2, 1, 0),
  initial.state = 30L)

# Cliff Walking (Sutton & Barto Example 6.6)
env3 = makeEnvironment("gridworld", shape = c(4, 12), goal.states = 47L,
  cliff.states = 37:46, reward.step = -1, reward.cliff = -100,
  cliff.transition.states = 36L, initial.state = 36L)
}
