% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table_spec.R
\name{define}
\alias{define}
\title{Defines a column}
\usage{
define(
  x,
  vars,
  label = NULL,
  format = NULL,
  align = NULL,
  label_align = NULL,
  width = NULL,
  visible = TRUE,
  n = NULL,
  blank_after = FALSE,
  dedupe = FALSE,
  id_var = FALSE,
  page_wrap = FALSE,
  page_break = FALSE,
  indent = NULL,
  label_row = FALSE
)
}
\arguments{
\item{x}{The table spec.}

\item{vars}{The variable name or names to define a column for.  Names may
be quoted or unquoted.  If defining for multiple variables, 
pass them as a vector of names.  If you want to pass an R variable of names,
escape the values with double curly braces, i.e. \code{vars = {{myvar}}}.
The curly brace escape is useful when writing functions that construct
reports dynamically.}

\item{label}{The label to use for the column header.  If a label is assigned
to the label column attribute, it will be used as a default.  Otherwise,
the column name will be used.}

\item{format}{The format to use for the column data.  The format can 
be a string format, a formatting function, a lookup list, a user-defined
format, or a formatting list. 
All formatting is performed by the \code{\link[fmtr]{fapply}} function from
the \code{\link[fmtr]{fmtr}} package.  For 
a list of common formatting codes, see \link[fmtr]{FormattingStrings}.}

\item{align}{The column alignment.  Valid values are "left", "right", 
"center", and "centre".  By default, text columns will be left aligned
and numeric columns will be right aligned.}

\item{label_align}{How to align the header labels for this column.
Valid values are "left", "right", "center", and "centre".  By default, 
the label alignment will follow any alignment set on the column \code{align}
parameter.}

\item{width}{The width of the column in the specified units of measure.
The units of measure are specified on the \code{units} parameter of the
\code{\link{create_report}} function.  If no width is supplied, the
\code{\link{write_report}} function will assign a default width based on the 
width of the column data and the label.  \code{write_report} will not set a 
column width less than the width of the largest word in the data or label.
In other words, \code{write_report} will not break words.}

\item{visible}{Whether or not the column should be visible on the report.
This parameter can be used as a simple way to drop columns from the report.}

\item{n}{The n value to place in the "N=" header label.  Formatting for
the n value will be performed by the formatting function assigned to the 
\code{n_format} parameter on \code{\link{create_table}}.}

\item{blank_after}{Whether to place a blank row after unique values of this
variable.  Valid values are TRUE or FALSE.  Default is FALSE.}

\item{dedupe}{Whether to dedupe the values for this variable.  Variables
that are deduped only show the value on the first row of each group.  This 
option is commonly used for grouping variables.}

\item{id_var}{Whether this variable should be considered an ID variable.
ID variables are retained on each page when the page is wrapped. ID variables
are also moved to the far left of the page.}

\item{page_wrap}{Force a page wrap on this variable.  A page wrap is a vertical
page break necessary when the table is too wide to fit on a single page.
The excess variables will be wrapped to the next page.  Page wraps will
continue until all columns are displayed.  Use the \code{id_var}
parameter to identify rows across wrapped pages.}

\item{page_break}{You may control when page breaks occur by defining
a page break variable yourself, and setting this parameter to TRUE for
that variable.  Only one page break variable can be defined per table.
If two or more variables are defined as a page break, an error will be 
generated.}

\item{indent}{How much to indent the column values.  The parameter takes a 
numeric value that will be interpreted according to the \code{units} 
(Unit Of Measure) setting on the report.  This parameter can be used to 
help create a stub column.  The default value is NULL, meaning the column
should not be indented.  See the \code{\link{stub}} function for additional
information on creating a stub column.}

\item{label_row}{Whether the values of the variable should be used to
create a label row.  Valid values are TRUE or FALSE.  Default is FALSE.
If \code{label_row} is set to TRUE, the dedupe parameter will also be 
set to TRUE.  This parameter is often used in conjunction with the 
\code{\link{stub}} function and \code{indent} parameter to create a 
stub column.}
}
\value{
The modified table spec.
}
\description{
A function to define a table column.  The
\code{define} function contains a variety of a parameters to control the 
appearance of the report.  Using the \code{define} function, you can control
simple options like column alignment and width, but also control more 
sophisticated options like page wrapping and page breaking.
}
\details{
Column definitions are optional.  By default, all columns in the data
are displayed in the order assigned to the data frame. 

The report will use attributes assigned to the data frame 
such as 'width', 'justify', 'label', and 'format'.  In other words, 
some control over the column 
formatting is available by manipulating the data frame attributes prior
to assigning the data frame to \code{create_table}.  See 
\code{\link{create_table}} for more details.

The \code{define} function is used to provide additional control over
column appearance.  For example, you may use the \code{define} function
to assign an "N=" population count, eliminate duplicates from the column,
or place a blank row after each unique value of the variable. 
See the parameter documentation for additional options.

Some of the parameters on the \code{define} function are used in the 
creation of a table stub.  Specifically, the \code{label_row} and 
\code{indent} parameters participate in the creation of the stub column.
See the \code{\link{stub}} function for further
information.

A single column definition may be defined for multiple variables.  
To create a definition for multiple variables, pass the variables as
a quoted or unquoted vector.  When creating a single definition for 
multiple variables, the parameters will be unified across those variables.
Note that some parameters (such as \code{page_break}) may only be set
once per report, and cannot be shared across multiple variables.
}
\examples{
library(reporter)
library(magrittr)
 
# Create temp file name
tmp <- file.path(tempdir(), "mtcars.txt")

# Prepare data
dat <- mtcars[1:10, ]
dat <- data.frame(vehicle = rownames(dat), dat)

# Define table
tbl <- create_table(dat, show_cols = 1:8) \%>\% 
  define(vehicle, label = "Vehicle", width = 3, id_var = TRUE, align = "left") \%>\% 
  define(mpg, label = "Miles per Gallon", width = 1) \%>\% 
  define(cyl, label = "Cylinders", format = "\%.1f") \%>\% 
  define(disp, label = "Displacement") \%>\% 
  define(hp, label = "Horsepower", page_wrap = TRUE) \%>\% 
  define(drat, visible = FALSE) \%>\% 
  define(wt, label = "Weight") \%>\% 
  define(qsec, label = "Quarter Mile Time", width = 1.5) 


# Create the report
rpt <- create_report(tmp, orientation = "portrait") \%>\% 
  titles("Listing 2.0", "MTCARS Data Listing with Page Wrap") \%>\% 
  add_content(tbl, align = "left") \%>\% 
  page_footer(right = "Page [pg] of [tpg]")

# Write the report
write_report(rpt)

# Send report to console for viewing
writeLines(readLines(tmp, encoding = "UTF-8"))

#                                  Listing 2.0
#                       MTCARS Data Listing with Page Wrap
# 
#                                         Miles per
# Vehicle                                    Gallon Cylinders Displacement
# ------------------------------------------------------------------------
# Mazda RX4                                      21       6.0          160
# Mazda RX4 Wag                                  21       6.0          160
# Datsun 710                                   22.8       4.0          108
# Hornet 4 Drive                               21.4       6.0          258
# Hornet Sportabout                            18.7       8.0          360
# Valiant                                      18.1       6.0          225
# Duster 360                                   14.3       8.0          360
# Merc 240D                                    24.4       4.0        146.7
# Merc 230                                     22.8       4.0        140.8
# Merc 280                                     19.2       6.0        167.6
# 
# ...
# 
#                                                                    Page 1 of 2
#                                  Listing 2.0
#                       MTCARS Data Listing with Page Wrap
# 
# Vehicle                              Horsepower Weight  Quarter Mile Time
# -------------------------------------------------------------------------
# Mazda RX4                                   110   2.62              16.46
# Mazda RX4 Wag                               110  2.875              17.02
# Datsun 710                                   93   2.32              18.61
# Hornet 4 Drive                              110  3.215              19.44
# Hornet Sportabout                           175   3.44              17.02
# Valiant                                     105   3.46              20.22
# Duster 360                                  245   3.57              15.84
# Merc 240D                                    62   3.19                 20
# Merc 230                                     95   3.15               22.9
# Merc 280                                    123   3.44               18.3
# 
# ...
# 
#                                                                    Page 2 of 2
}
\seealso{
Other table: 
\code{\link{column_defaults}()},
\code{\link{create_table}()},
\code{\link{print.table_spec}()},
\code{\link{spanning_header}()},
\code{\link{stub}()}
}
\concept{table}
