% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/postProcess.R
\name{postProcess}
\alias{postProcess}
\alias{postProcess.list}
\alias{postProcess.default}
\title{Generic function to post process objects}
\usage{
postProcess(x, ...)

\method{postProcess}{list}(x, ...)

\method{postProcess}{default}(x, ...)
}
\arguments{
\item{x}{A GIS object of postProcessing,
e.g., Spat* or sf*. This can be provided as a
\code{rlang::quosure} or a normal R object.}

\item{...}{Additional arguments passed to methods. For \code{spatialClasses},
these are: \code{\link[=cropTo]{cropTo()}}, \code{\link[=fixErrorsIn]{fixErrorsIn()}},
\code{\link[=projectTo]{projectTo()}}, \code{\link[=maskTo]{maskTo()}},
\code{\link[=determineFilename]{determineFilename()}}, and \code{\link[=writeTo]{writeTo()}}.
Each of these may also pass \code{...} into other functions, like
\code{\link[=writeTo]{writeTo()}}.
This might include potentially important arguments like \code{datatype},
\code{format}. Also passed to \code{terra::project},
with likely important arguments such as \code{method = "bilinear"}.
See details.}
}
\value{
A GIS file (e.g., \code{RasterLayer}, \code{SpatRaster} etc.) that has been
appropriately cropped, reprojected, masked, depending on the inputs.
}
\description{
\if{html}{\figure{lifecycle-maturing.svg}{options: alt="maturing"}}

The method for GIS objects (terra \verb{Spat*} & sf classes) will
crop, reproject, and mask, in that order.
This is a wrapper for \code{\link[=cropTo]{cropTo()}}, \code{\link[=fixErrorsIn]{fixErrorsIn()}},
\code{\link[=projectTo]{projectTo()}}, \code{\link[=maskTo]{maskTo()}} and \code{\link[=writeTo]{writeTo()}},
with a required amount of data manipulation between these calls so that the crs match.
}
\section{Post processing sequence}{


If the \code{rasterToMatch} or \code{studyArea} are passed, then
the following sequence will occur:

\enumerate{
\item Fix errors \code{\link[=fixErrorsIn]{fixErrorsIn()}}. Currently only errors fixed are for
\code{SpatialPolygons} using \code{buffer(..., width = 0)}.
\item Crop using \code{\link[=cropTo]{cropTo()}}
\item Project using \code{\link[=projectTo]{projectTo()}}
\item Mask using \code{\link[=maskTo]{maskTo()}}
\item Determine file name \code{\link[=determineFilename]{determineFilename()}}
\item Write that file name to disk, optionally \code{\link[=writeTo]{writeTo()}}
}

NOTE: checksumming does not occur during the post-processing stage, as
there are no file downloads. To achieve fast results, wrap
\code{prepInputs} with \code{Cache}
}

\section{Backwards compatibility with \code{rasterToMatch} and/or \code{studyArea} arguments}{


For backwards compatibility, \code{postProcess} will continue to allow passing
\code{rasterToMatch} and/or \code{studyArea} arguments. Depending on which of these
are passed, different things will happen to the \code{targetFile} located at \code{filename1}.

See \emph{Use cases} section in \code{\link[=postProcessTo]{postProcessTo()}} for post processing behaviour with
the new \code{from} and \code{to} arguments.

\subsection{If \code{targetFile} is a raster (\verb{Raster*}, or \code{SpatRaster}) object:}{
\tabular{lccc}{
\tab \code{rasterToMatch}      \tab \code{studyArea}  \tab  Both               \cr
\code{extent}     \tab Yes                  \tab   Yes        \tab \code{rasterToMatch}     \cr
\code{resolution} \tab Yes                  \tab   No         \tab \code{rasterToMatch}     \cr
\code{projection} \tab Yes                  \tab   No*        \tab \code{rasterToMatch}*    \cr
\code{alignment}  \tab Yes                  \tab   No         \tab \code{rasterToMatch}     \cr
\code{mask}       \tab No**                 \tab   Yes        \tab \code{studyArea}**       \cr
}
*Can be overridden with \code{useSAcrs}.

**Will mask with \code{NA}s from \code{rasterToMatch} if \code{maskWithRTM}.
}

\subsection{If \code{targetFile} is a vector (\verb{Spatial*}, \code{sf} or \code{SpatVector}) object:}{
\tabular{lccc}{
\tab \code{rasterToMatch}      \tab \code{studyArea}  \tab Both               \cr
\code{extent}     \tab Yes                  \tab   Yes        \tab \code{rasterToMatch}    \cr
\code{resolution} \tab NA                   \tab   NA         \tab NA                 \cr
\code{projection} \tab Yes                  \tab   No*        \tab \code{rasterToMatch}*   \cr
\code{alignment}  \tab NA                   \tab   NA         \tab NA                 \cr
\code{mask}       \tab No                   \tab   Yes        \tab \code{studyArea}        \cr
}
*Can be overridden with \code{useSAcrs}
}
}

\examples{
if (requireNamespace("terra", quietly = TRUE) && requireNamespace("sf", quietly = TRUE)) {
  library(reproducible)
  od <- setwd(tempdir2())
  # download a (spatial) file from remote url (which often is an archive) load into R
  # need 3 files for this example; 1 from remote, 2 local
  dPath <- file.path(tempdir2())
  remoteTifUrl <- "https://github.com/rspatial/terra/raw/master/inst/ex/elev.tif"

  localFileLuxSm <- system.file("ex/luxSmall.shp", package = "reproducible")
  localFileLux <- system.file("ex/lux.shp", package = "terra")

  # 1 step for each layer
  # 1st step -- get study area
  studyArea <- prepInputs(localFileLuxSm, fun = "terra::vect") # default is sf::st_read

  # 2nd step: make the input data layer like the studyArea map
  # Test only relevant if connected to internet -- so using try just in case
  elevForStudy <- try(prepInputs(url = remoteTifUrl, to = studyArea, res = 250,
                             destinationPath = dPath))

  # Alternate way, one step at a time. Must know each of these steps, and perform for each layer
  \donttest{
    dir.create(dPath, recursive = TRUE, showWarnings = FALSE)
    file.copy(localFileLuxSm, file.path(dPath, basename(localFileLuxSm)))
    studyArea2 <- terra::vect(localFileLuxSm)
    if (!all(terra::is.valid(studyArea2))) studyArea2 <- terra::makeValid(studyArea2)
    tf <- tempfile(fileext = ".tif")
    download.file(url = remoteTifUrl, destfile = tf, mode = "wb")
    Checksums(dPath, write = TRUE, files = tf)
    elevOrig <- terra::rast(tf)
    elevForStudy2 <- terra::project(elevOrig, terra::crs(studyArea2), res = 250) |>
      terra::crop(studyArea2) |>
      terra::mask(studyArea2)

    isTRUE(all.equal(studyArea, studyArea2)) # Yes!
  }

  # sf class
  studyAreaSmall <- prepInputs(localFileLuxSm)
  studyAreas <- list()
  studyAreas[["orig"]] <- prepInputs(localFileLux)
  studyAreas[["reprojected"]] <- projectTo(studyAreas[["orig"]], studyAreaSmall)
  studyAreas[["cropped"]] <- suppressWarnings(cropTo(studyAreas[["orig"]], studyAreaSmall))
  studyAreas[["masked"]] <- suppressWarnings(maskTo(studyAreas[["orig"]], studyAreaSmall))

  # SpatVector-- note: doesn't matter what class the "to" object is, only the "from"
  studyAreas <- list()
  studyAreas[["orig"]] <- prepInputs(localFileLux, fun = "terra::vect")
  studyAreas[["reprojected"]] <- projectTo(studyAreas[["orig"]], studyAreaSmall)
  studyAreas[["cropped"]] <- suppressWarnings(cropTo(studyAreas[["orig"]], studyAreaSmall))
  studyAreas[["masked"]] <- suppressWarnings(maskTo(studyAreas[["orig"]], studyAreaSmall))
  if (interactive()) {
    par(mfrow = c(2,2));
    out <- lapply(studyAreas, function(x) terra::plot(x))
  }

  setwd(od)
}
}
\seealso{
\code{prepInputs}
}
