% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CSC.R
\name{CSC}
\alias{CSC}
\title{Cause-specific Cox proportional hazard regression}
\usage{
CSC(formula, data, cause, survtype = "hazard", fitter = "coxph",
  iid = TRUE, ...)
}
\arguments{
\item{formula}{Either a single \code{Hist} formula or a list of formulas. 
If it is a list it must contain as many \code{Hist} formulas as there are
causes when \code{survtype="hazard"} and exactly two formulas when \code{survtype="survival"}.
If it is a list the first formula is used for the cause of interest specific Cox regression
and the other formula(s) either for the other cause specific Cox regression(s) or for the
Cox regression of the combined event where each cause counts as event. Note that when only one
formula is given the covariates enter in exactly the same way into all Cox regression analyses.}

\item{data}{A data in which to fit the models.}

\item{cause}{The cause of interest. Defaults to the first cause (see Details).}

\item{survtype}{Either \code{"hazard"} (the default) or
\code{"survival"}.  If \code{"hazard"} fit cause-specific Cox
regression models for all causes.  If \code{"survival"} fit one
cause-specific Cox regression model for the cause of interest and
also a Cox regression model for event-free survival.}

\item{fitter}{Routine to fit the Cox regression models.
If \code{coxph} use \code{survival::coxph} else use \code{rms::cph}.}

\item{iid}{logical. Compute the value of the influence function for each Cox model and store it in the (cause-specific) Cox regression object.}

\item{...}{Arguments given to \code{coxph}.}
}
\value{
\item{models }{a list with the fitted (cause-specific) Cox
regression objects} \item{response }{the event history response }
\item{eventTimes }{the sorted (unique) event times } \item{survtype }{the
value of \code{survtype}} \item{theCause }{the cause of interest. see
\code{cause}} \item{causes }{the other causes} %% ...
}
\description{
Interface for fitting cause-specific Cox proportional hazard regression
models in competing risk.
}
\details{
The causes and their order
are determined by \code{prodlim::getStates()} applied to the Hist object.
}
\examples{

library(prodlim)
library(pec)
library(survival)
data(Melanoma)
## fit two cause-specific Cox models
## different formula for the two causes
fit1 <- CSC(list(Hist(time,status)~sex,Hist(time,status)~invasion+epicel+age),
            data=Melanoma, iid = TRUE)
print(fit1)



## model hazard of all cause mortality instead of hazard of type 2 
fit1a <- CSC(list(Hist(time,status)~sex+age,Hist(time,status)~invasion+epicel+log(thick)),
             data=Melanoma,
             survtype="surv")
print(fit1a)

## special case where cause 2 has no covariates
fit1b <- CSC(list(Hist(time,status)~sex+age,Hist(time,status)~1),
             data=Melanoma)
print(fit1b)
predict(fit1b,cause=1,times=100,newdata=Melanoma)


## same formula for both causes
fit2 <- CSC(Hist(time,status)~invasion+epicel+age,
            data=Melanoma)
print(fit2)

## combine a cause-specific Cox regression model for cause 2 
## and a Cox regression model for the event-free survival:
## different formula for cause 2 and event-free survival
fit3 <- CSC(list(Hist(time,status)~sex+invasion+epicel+age,
                 Hist(time,status)~invasion+epicel+age),
            survtype="surv",
            data=Melanoma)
print(fit3)

## same formula for both causes
fit4 <- CSC(Hist(time,status)~invasion+epicel+age,
            data=Melanoma,
            survtype="surv")
print(fit4)

## strata
fit5 <- CSC(Hist(time,status)~invasion+epicel+age+strata(sex),
            data=Melanoma,
            survtype="surv")
print(fit5)

## sanity checks

cox1 <- coxph(Surv(time,status==1)~invasion+epicel+age+strata(sex),data=Melanoma)
cox2 <- coxph(Surv(time,status!=0)~invasion+epicel+age+strata(sex),data=Melanoma)
all.equal(cox1,fit5$models[[1]])
all.equal(cox2,fit5$models[[2]])

## predictions
##
## survtype = "hazard": predictions for both causes can be extracted
## from the same fit
library(pec)
fit2 <- CSC(Hist(time,status)~invasion+epicel+age, data=Melanoma)
predict(fit2,cause=1,newdata=Melanoma[c(17,99,108),],times=c(100,1000,10000))
predictRisk(fit2,cause=1,newdata=Melanoma[c(17,99,108),],times=c(100,1000,10000))
predictRisk(fit2,cause=2,newdata=Melanoma[c(17,99,108),],times=c(100,1000,10000))
predict(fit2,cause=1,newdata=Melanoma[c(17,99,108),],times=c(100,1000,10000))
predict(fit2,cause=2,newdata=Melanoma[c(17,99,108),],times=c(100,1000,10000))

## survtype = "surv" we need to change the cause of interest 
library(survival)
fit5.2 <- CSC(Hist(time,status)~invasion+epicel+age+strata(sex),
            data=Melanoma,
            survtype="surv",cause=2)
## now this does not work
try(predictRisk(fit5.2,cause=1,newdata=Melanoma,times=4))

## but this does
predictRisk(fit5.2,cause=2,newdata=Melanoma,times=100)
predict(fit5.2,cause=2,newdata=Melanoma,times=100)
predict(fit5.2,cause=2,newdata=Melanoma[4,],times=100)

}
\author{
Thomas A. Gerds \email{tag@biostat.ku.dk} and Ulla B. Mogensen
}
\references{
J Benichou and Mitchell H Gail. Estimates of absolute cause-specific risk
in cohort studies. Biometrics, pages 813--826, 1990.

T.A. Gerds, T.H. Scheike, and P.K. Andersen. Absolute risk regression for
competing risks: interpretation, link functions, and prediction. Statistics
in Medicine, 31(29):3921--3930, 2012.
}
\seealso{
\code{\link{coxph}}
}
\keyword{survival}

