% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ghcnd.R
\name{ghcnd}
\alias{ghcnd}
\title{Get all GHCND data from a single weather site}
\usage{
ghcnd(stationid, ...)
}
\arguments{
\item{stationid}{A character string giving the identification of the weather
station for which the user would like to pull data. To get a full and
current list of stations, the user can use the \code{\link{ghcnd_stations}}
function. To identify stations within a certain radius of a location, the
user can use the \code{\link{meteo_nearby_stations}} function.}

\item{...}{Additional curl options to pass through to \code{\link[httr]{GET}}.}
}
\value{
A tibble (data.frame) which contains data pulled from NOAA's FTP
server for the queried weather site. A README file with more information
about the format of this file is available from NOAA
(\url{http://www1.ncdc.noaa.gov/pub/data/ghcn/daily/readme.txt}).
This file is formatted so each line of the file gives the daily weather
observations for a single weather variable for all days of one month of
one year. In addition to measurements, columns are included for certain
flags, which add information on observation sources and quality and are
further explained in NOAA's README file for the data.
}
\description{
This function uses ftp to access the Global Historical Climatology Network
daily weather data from NOAA's FTP server for a single weather site. It
requires the site identification number for that site and will pull the
entire weather dataset for the site.
}
\details{
This function saves the full set of weather data for the queried
site locally in the directory specified by the \code{path} argument.

You can access the path for the cached file via \code{attr(x, "source")}
}
\section{File storage}{

We use \pkg{rappdirs} to store files, see
\code{\link[rappdirs]{user_cache_dir}} for how we determine the directory on
your machine to save files to, and run
\code{rappdirs::user_cache_dir("rnoaa/ghcnd")} to get that directory.

Note that between versions of \pkg{rnoaa} you may want to clear your
cache of ghcnd files IF there are changes in ghcnd functions. See
\code{\link{ghcnd_clear_cache}} or you can do so manually.
}

\examples{
\dontrun{
# Get data
ghcnd(stationid = "AGE00147704")

stations <- ghcnd_stations()
ghcnd(stations$id[40])
ghcnd(stations$id[4000])
ghcnd(stations$id[10000])
ghcnd(stations$id[80000])
ghcnd(stations$id[80300])

library("dplyr")
ghcnd(stations$id[80300]) \%>\% select(id, element) \%>\% slice(1:3)

# manipulate data
## using built in fxns
dat <- ghcnd(stationid = "AGE00147704")
(alldat <- ghcnd_splitvars(dat))
library("ggplot2")
ggplot(subset(alldat$tmax, tmax >= 0), aes(date, tmax)) + geom_point()

## using dplyr
library("dplyr")
dat <- ghcnd(stationid = "AGE00147704")
dat \%>\%
 filter(element == "PRCP", year == 1909)
}
}
\seealso{
To generate a weather dataset for a single weather site that has
been cleaned to a tidier weather format, the user should use the
\code{\link{ghcnd_search}} function, which calls \code{\link{ghcnd}} and then
processes the output, or \code{\link{meteo_tidy_ghcnd}}, which wraps the
\code{\link{ghcnd_search}} function to output a tidy dataframe. To pull
GHCND data from multiple monitors, see \code{\link{meteo_pull_monitors}}.
}
\author{
Scott Chamberlain \email{myrmecocystus@gmail.com},
Adam Erickson \email{adam.erickson@ubc.ca}
}
