% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cortest.R
\name{cortest}
\alias{cortest}
\title{Calibrated tests for correlation between paired samples}
\usage{
cortest(
  x,
  y,
  alternative = "two.sided",
  method = "pearson",
  ties.break = "none",
  conf.level = 0.95
)
}
\arguments{
\item{x, y}{the two continuous variables. Must be of same length.}

\item{alternative}{indicates the alternative hypothesis and must be one of "two.sided", "greater" or "less".}

\item{method}{a character string indicating which test to implement. Can be \code{"pearson"}, \code{"kendall"} or \code{"spearman"}.}

\item{ties.break}{the method used to break ties in case there are ties in the x or y vectors. Can be \code{"none"} or \code{"random"}.}

\item{conf.level}{confidence level for the confidence interval of the correlation coefficient. It is used only for the Pearson's correlation test.}
}
\value{
Returns the result of the test with its corresponding p-value, the value of the test statistic and the estimated value of Pearson's correlation coefficient,
Kendall's tau or Spearman's rho. For Pearson's correlation test an asymptotic confidence interval for the correlation coefficient is also returned.
}
\description{
Tests the association/correlation for continuous paired samples using corrected versions of Pearson's, Kendall's and Spearman's correlation tests. These three tests are asymptotically well calibrated.
}
\details{
Three tests are implemented. The null hypothesis for the corrected Pearson test is: H0 Cor(X,Y)=0 where Cor represents Pearson's correlation coefficient.
The alternative is specified by the \code{alternative} argument. The null hypothesis for the corrected Kendall test is: H0 tau=0 where tau represents Kendall's tau coefficient.
The null hypothesis for the corrected Spearman test is: H0 rho=0 where rho represents Spearman's rho coefficient.
All tests are asymptotically well calibrated in the sense that the rejection probability under the null hypothesis is asymptotically equal to the level of the test.
For the Pearson test, the exact distribution of the test statistic under the Gaussian case has been tabulated for n<130. For n>=130, the Student distribution with n-2 degrees of
freedom is used.

When Pearson's correlation test is used, a confidence interval for Pearson's correlation coefficient is also returned. This confidence interval has been implemented
from the delta-method. It should be noted that this method is asymptotic and can display very narrow intervals for small sample sizes and thus can suffer from
low coverage probabilities. We therefore recommend to use confidence intervals for Pearson's correlation coefficient only when n is at least larger than 100.

The Kendall and Spearman correlation tests are not valid in the presence of ties in the \code{x} or \code{y} vector. If ties occur, they should be broken using the
\code{ties.break} option. Note that they can also be broken outside \code{cortest} using the function \code{tiebreak}.
}
\note{
The option \code{ties.break} handles ties for both Kendall's and Spearman's test. If \code{ties.break="none"} the ties are ignored, if \code{ties.break="random"} they are randomly broken.
Note that only ties inside each vector are broken (but not ties between the two vectors).
}
\examples{
#Application on the Evans dataset
#Description of this dataset is available in the lbreg package
data(Evans)
with(Evans,cor.test(CHL[CDH==1],DBP[CDH==1]))
with(Evans,cortest(CHL[CDH==1],DBP[CDH==1]))
#The pvalues are very different!

with(Evans,cortest(CHL[CDH==1],DBP[CDH==1],method="kendall",ties.break="random"))
with(Evans,cortest(CHL[CDH==1],DBP[CDH==1],method="spearman",ties.break="random"))

#We use the function tiebreak to remove ties and compare the results from cor.test with cortest
X=tiebreak(Evans$CHL[Evans$CDH==1])
Y=tiebreak(Evans$DBP[Evans$CDH==1])
cor.test(X,Y,method="kendall")
cortest(X,Y,method="kendall")
cor.test(X,Y,method="spearman")
cortest(X,Y,method="spearman")

\donttest{
#Simulated data
n=100 #sample size
M=10000 #number of replications
testone=function(n){
X=rnorm(n,0,1)
epsi=rnorm(n,0,1)
Y=X^2+0.3*epsi
list(test1=cortest(X,Y)$p.value,test2=cor.test(X,Y)$p.value) #cor.test is the standard Pearson test
}
res1=res2=rep(NA,M)
# Replications in order to check if the the corrected Pearson test and
# the standard test are well calibrated
for (i in 1:M)
{
result=testone(n)
res1[i]=result$test1
res2[i]=result$test2
}
mean(res1<0.05)
mean(res2<0.05)}

\donttest{
#Replications with Kendall's test (may take a few minutes to run)
M=500
testone=function(n){
X=rnorm(n,0,1)
epsi=rnorm(n,0,1)
Y=X^2+0.3*epsi
list(test1=cortest(X,Y)$p.value,test2=cor.test(X,Y)$p.value,
test3=cortest(X,Y,method="kendall")$p.value,
test4=cor.test(X,Y,method="kendall")$p.value,
test5=cortest(X,Y,method="spearman")$p.value,
test6=cor.test(X,Y,method="spearman")$p.value)
#cor.test is the standard Pearson, Kendall or Spearman correlation test
}
res1=res2=res3=res4=res5=res6=rep(NA,M)
# Replications to check if the tests are well calibrated
for (i in 1:M)
{
result=testone(n)
res1[i]=result$test1
res2[i]=result$test2
res3[i]=result$test3
res4[i]=result$test4
res5[i]=result$test5
res6[i]=result$test6
}
mean(res1<0.05)
mean(res2<0.05)
mean(res3<0.05)
mean(res4<0.05)
mean(res5<0.05)
mean(res6<0.05)}
}
\seealso{
\code{\link{vartest}}, \code{\link{indeptest}}, \code{\link{mediantest}}, \code{\link{wilcoxtest}}, \code{\link{tiebreak}}.
}
\keyword{test}
