% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_WRMF.R
\docType{data}
\name{WRMF}
\alias{WRMF}
\title{Weighted Regularized Matrix Facrtorization for collaborative filtering}
\format{\code{R6Class} object.}
\usage{
WRMF
}
\description{
Creates matrix factorization model which could be solved with Alternating Least Squares (Weighted ALS for implicit feedback).
For implicit feedback see (Hu, Koren, Volinsky)'2008 paper \url{http://yifanhu.net/PUB/cf.pdf}.
For explicit feedback model is classic model for rating matrix decomposition with MSE error (without biases at the moment).
These two algorithms are proven to work well in recommender systems.
}
\section{Usage}{

For usage details see \bold{Methods, Arguments and Examples} sections.
\preformatted{
  model = WRMF$new(rank = 10L, lambda = 0,
                  feedback = c("implicit", "explicit"),
                  non_negative = FALSE,
                  solver = c("conjugate_gradient", "cholesky"),
                  cg_steps = 3L,
                  init = NULL)
  model$fit_transform(x, n_iter = 5L, ...)
  model$transform(x)
  model$predict(x, k, not_recommend = x, items_exclude = NULL, ...)
  model$components
  model$remove_scorer(name)
}
}

\section{Methods}{

\describe{
  \item{\code{$new(rank = 10L, lambda = 0, feedback = c("implicit", "explicit"),
                   non_negative = FALSE,
                   solver = c("conjugate_gradient", "cholesky"), cg_steps = 3L,
                   init = NULL) }}{ creates matrix
    factorization model model with \code{rank} latent factors. If \code{init} is provided then initialize
    item embeddings with its values.}
  \item{\code{$fit_transform(x, n_iter = 5L, ...)}}{ fits model to
    an input user-item matrix. (preferably in "dgCMatrix" format).
    For implicit feedback \code{x} should be a confidence matrix which corresponds to \code{1 + alpha * r_ui} in original paper.
    Usually \code{r_ui} corresponds to the number of interactions of user \code{u} and item \code{i}.
    For explicit feedback values in \code{x} represents ratings.
    \bold{Returns factor matrix for users of size \code{n_users * rank}}}
  \item{\code{$transform(x, ...)}}{Calculates user embeddings from given \code{x} user-item matrix.
    Result is \code{n_users * rank} matrix}
  \item{\code{$predict(x, k, not_recommend = x, ...)}}{predicts \code{top k}
    item indices for users \code{x}. Additionally contains \code{scores} attribute - "score"
    values for each prediction. If model contains item ids (input matrix to \code{fit_transform()} had column-names
    then result additionally will have \code{ids} attribute - item ids which
    correspond to item indices.
    Users features \code{x} should be defined the same way as they were defined in training data -
    as \bold{sparse matrix} of confidence values (implicit feedback) or ratings (explicit feedback).
    Column names (=item ids) should be in the same order as in the \code{fit_transform()}.}
  \item{\code{$components}}{items embeddings matrix of size \code{rank * n_items}}
}
}

\section{Arguments}{

\describe{
 \item{model}{A \code{WRMF} model.}
 \item{x}{An input sparse user-item matrix(of class \code{dgCMatrix}).
 For explicit feedback should consists of ratings.
 For implicit feedback all positive interactions should be filled with \bold{confidence} values.
 Missed interactions should me zeros/empty.
 So for simple case case when \code{confidence = 1 + alpha * x}}
 \item{rank}{\code{integer} - number of latent factors}
 \item{lambda}{\code{numeric} - regularization parameter}
 \item{feedback}{\code{character} - feedback type - one of \code{c("implicit", "explicit")}}
 \item{solver}{\code{character} - solver for "implicit feedback" problem.
    One of \code{c("conjugate_gradient", "cholesky")}.
    Usually approximate \code{"conjugate_gradient"} is significantly faster and solution is
    on par with exact \code{"cholesky"}}
 \item{cg_steps}{\code{integer > 0} - max number of internal steps in conjugate gradient
    (if "conjugate_gradient" solver used). \code{cg_steps = 3} by default.
    Controls precision of linear equation solution at the each ALS step. Usually no need to tune this parameter.}
 \item{preprocess}{\code{function} = \code{identity()} by default. User spectified function which will be applied to user-item interaction matrix
    before running matrix factorization (also applied in inference time before making predictions). For example we may
    want to normalize each row of user-item matrix to have 1 norm. Or apply \code{log1p()} to discount large counts.
    This essentially corresponds to the "confidence" function from (Hu, Koren, Volinsky)'2008 paper \url{http://yifanhu.net/PUB/cf.pdf}}
 \item{precision}{one of \code{c("double", "float")}. Should embeeding matrices be usual numeric or
 float (from \code{float} package). The latter is usually 2x faster and consumes less RAM. BUT \code{float} matrices
 are not "base" objects. Use carefully.}
 \item{not_recommend}{\code{sparse matrix} or \code{NULL} - points which items should be excluided from recommendations for a user.
   By default it excludes previously seen/consumed items.}
 \item{items_exclude}{\code{character} = item ids or \code{integer} = item indices or \code{NULL} -
 items to exclude from recommendations for \bold{all} users.}
 \item{convergence_tol}{{\code{numeric = -Inf} defines early stopping strategy. Model stops fitting
    when one of two following conditions is satisfied: (a) exceed number of iterations,
    or (b) \code{loss_previous_iter / loss_current_iter - 1 < convergence_tol}}}
 \item{...}{other arguments. Not used at the moment}
}
}

\examples{
data('movielens100k')
train = movielens100k[1:900, ]
cv = movielens100k[901:nrow(movielens100k), ]
model = WRMF$new(rank = 5,  lambda = 0, feedback = 'implicit')
user_emb = model$fit_transform(train, n_iter = 5, convergence_tol = -1)
item_emb = model$components
preds = model$predict(cv, k = 10, not_recommend = cv)
}
\seealso{
\itemize{
  \item{\url{https://math.stackexchange.com/questions/1072451/analytic-solution-for-matrix-factorization-using-alternating-least-squares/1073170#1073170}}
  \item{\url{http://activisiongamescience.github.io/2016/01/11/Implicit-Recommender-Systems-Biased-Matrix-Factorization/}}
  \item{\url{http://datamusing.info/blog/2015/01/07/implicit-feedback-and-collaborative-filtering/}}
  \item{\url{https://jessesw.com/Rec-System/}}
  \item{\url{http://danielnee.com/2016/09/collaborative-filtering-using-alternating-least-squares/}}
  \item{\url{http://www.benfrederickson.com/matrix-factorization/}}
  \item{\url{http://www.benfrederickson.com/fast-implicit-matrix-factorization/}}
}
}
\keyword{datasets}
