% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prior_summary.R
\name{prior_summary.stanreg}
\alias{prior_summary}
\alias{prior_summary.stanreg}
\title{Summarize the priors used for an rstanarm model}
\usage{
\method{prior_summary}{stanreg}(object, digits = 2, ...)
}
\arguments{
\item{object}{A fitted model object returned by one of the 
\pkg{rstanarm} modeling functions. See \code{\link{stanreg-objects}}.}

\item{digits}{Number of digits to use for rounding.}

\item{...}{Currently ignored by the method for stanreg objects. The S3
generic uses \code{...} to pass arguments to any defined methods.}
}
\value{
A list of class "prior_summary.stanreg", which has its own print
  method.
}
\description{
The \code{prior_summary} method provides a summary of the prior distributions
used for the parameters in a given model. In some cases the user-specified
prior does not correspond exactly to the prior used internally by
\pkg{rstanarm} (see the sections below). Especially in these cases, but also
in general, it can be much more useful to visualize the priors. Visualizing
the priors can be done using the \code{\link{posterior_vs_prior}} function,
or alternatively by fitting the model with the \code{prior_PD} argument set
to \code{TRUE} (to draw from the prior predictive distribution instead of
conditioning on the outcome) and then plotting the parameters.
}
\section{Intercept (after predictors centered)}{
 
  For \pkg{rstanarm} modeling functions that accept a \code{prior_intercept} 
  argument, the specified prior for the intercept term applies to the 
  intercept after \pkg{rstanarm} internally centers the predictors so they 
  each have mean zero. The estimate of the intercept returned to the user 
  correspond to the intercept with the predictors as specified by the user 
  (unmodified by \pkg{rstanarm}), but when \emph{specifying} the prior the 
  intercept can be thought of as the expected outcome when the predictors are
  set to their means. The only exception to this is for models fit with the 
  \code{sparse} argument set to \code{TRUE} (which is only possible with a
  subset of the modeling functions and never the default).
}

\section{Adjusted scales}{
 For some models you may see "\code{adjusted scale}"
  in the printed output and adjusted scales included in the object returned 
  by \code{prior_summary}. These adjusted scale values are the prior scales 
  actually used by \pkg{rstanarm} and are computed by adjusting the prior 
  scales specified by the user to account for the scales of the predictors 
  (as described in the documentation for the \code{\link[=priors]{autoscale}}
  argument). To disable internal prior scale adjustments set the 
  \code{autoscale} argument to \code{FALSE} when setting a prior using one of
  the distributions that accepts an \code{autoscale} argument. For example,
  \code{normal(0, 5, autoscale=FALSE)} instead of just \code{normal(0, 5)}.
}

\section{Coefficients in Q-space}{

  For the models fit with an \pkg{rstanarm} modeling function that supports 
  the \code{QR} argument (see e.g, \code{\link{stan_glm}}), if \code{QR} is 
  set to \code{TRUE} then the prior distributions for the regression
  coefficients specified using the \code{prior} argument are not relative to
  the original predictor variables \eqn{X} but rather to the variables in the
  matrix \eqn{Q} obtained from the \eqn{QR} decomposition of \eqn{X}. 
  
  In particular, if \code{prior = normal(location,scale)}, then this prior on
  the coefficients in \eqn{Q}-space can be easily translated into a joint 
  multivariate normal (MVN) prior on the coefficients on the original 
  predictors in \eqn{X}. Letting \eqn{\theta} denote the coefficients on
  \eqn{Q} and \eqn{\beta} the coefficients on \eqn{X} then if \eqn{\theta
  \sim N(\mu, \sigma)}{\theta ~ N(\mu, \sigma)} the corresponding prior on
  \eqn{\beta} is \eqn{\beta \sim MVN(R\mu, R'R\sigma^2)}{\beta ~ MVN(R\mu,
  R'R\sigma)}, where \eqn{\mu} and \eqn{\sigma} are vectors of the
  appropriate length. Technically \pkg{rstanarm} uses a scaled \eqn{QR}
  decomposition to ensure that the columns of the predictor matrix used to
  fit the model all have unit scale. The matrices actually used are
  \eqn{Q^\ast = Q \sqrt{n-1}}{Q* = Q (n-1)^0.5} and \eqn{R^\ast =
  \frac{1}{\sqrt{n-1}} R}{R* = (n-1)^(-0.5) R} (see the documentation for the
  \code{\link[=stan_glm]{QR}} argument).
  
  If you are interested in the prior on \eqn{\beta} implied by the prior on
  \eqn{\theta}, we recommend visualizing it as described above in the
  \strong{Description} section, which is simpler than working it out
  analytically.
}
\examples{
if (!exists("example_model")) example(example_model) 
prior_summary(example_model)

priors <- prior_summary(example_model)
names(priors)
priors$prior$scale
priors$prior$adjusted_scale

# for a glm with adjusted scales (see Details, above), compare 
# the default (rstanarm adjusting the scales) to setting 
# autoscale=FALSE for prior on coefficients
fit <- stan_glm(mpg ~ wt + am, data = mtcars, 
                prior = normal(0, c(2.5, 4)), 
                prior_intercept = normal(0, 5), 
                iter = 10, chains = 1) # only for demonstration 
prior_summary(fit)

fit2 <- update(fit, prior = normal(0, c(2.5, 4), autoscale=FALSE), 
               prior_intercept = normal(0, 5, autoscale=FALSE))
prior_summary(fit2)

}
\seealso{
\code{\link{posterior_vs_prior}}, \code{\link{priors}}
}

