% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transform-helper-functions.R,
%   R/transform.scdf.R
\name{moving_median}
\alias{moving_median}
\alias{moving_mean}
\alias{local_regression}
\alias{first_of}
\alias{across_cases}
\alias{all_cases}
\alias{transform.scdf}
\title{Transform variables in every single case of a single case data frame}
\usage{
moving_median(x, lag = 1)

moving_mean(x, lag = 1)

local_regression(x, mt = 1:length(x), f = 0.2)

first_of(x, positions = 0)

across_cases(...)

all_cases(...)

\method{transform}{scdf}(`_data`, ...)
}
\arguments{
\item{x}{A logical vector.}

\item{lag}{Number of values surrounding a value to calculate the average}

\item{mt}{A vector with measurement times.}

\item{f}{the proportion of surrounding data influencing each data point.}

\item{positions}{A numeric vector with relative positions to the first 
appearance of a TRUE value in x.}

\item{...}{Expressions.}

\item{_data}{An scdf.}
}
\value{
An scdf.
}
\description{
Takes an scdf and applies transformations to each individual case. This is
useful to calculate or modify new variables.
}
\details{
This function is a method of the generic transform function. Unlike the
generic function, it calculates expressions serially. This means that the
results of the calculation of one expression are the basis for the following
computations. The \code{all_cases} function is a helper function that
extracts the values of a variable from all cases. It takes an expression as
an argument. For example, \code{mean(all_cases(values))} calculates the mean
of the values from all cases. \code{mean(all_cases(values[phase == "A"]))}
will calculate the mean of all values where phase is A. The function
\code{across_cases} allows to calculate new variables or replace existing
variables across all
cases. E.g., \code{across_cases(values_ranked = rank(values, na.last =
"keep"))} will calculate a new variable with values ranked across all cases.
}
\examples{
## Creates a single-case with frequency distributions. The proportion and
## percentage of the frequencies are calculated with transform:
design <- design(
 n = 3,
 level = 5,
 distribution = "binomial",
 n_trials = 20,
 start_value = 0.5
)
study <- random_scdf(design)
transform(study, proportion = values/trials, percentage = proportion * 100)

## Z standardize the dependent variable and add two new variables:
exampleAB \%>\%
  transform(
    values = scale(values),
    mean_values = mean(values),
    sd_values = sd(values)
  )

## Use `all` to calculate global variables.
exampleAB \%>\%
  transform(
    values_center_case = values - mean(values[phase == "A"]),
    values_center_global = values - mean(all(values[phase == "A"])),
    value_dif = values_center_case - values_center_global
  )

## Use `across_cases` to calculate or replace a variable with values from
## all cases. E.g., standardize the dependent variable:
exampleABC \%>\%
  transform(
    across_cases(values = scale(values))
  )

## Rank transform the values based on all cases vs. within each case:
exampleABC \%>\%
  transform(
    across_cases(values_across = rank(values, na.last="keep")),
    value_within = rank(values, na.last="keep")
  )

## Three helper functions to smooth the data
Huber2014$Berta \%>\%
transform(
  "compliance (moving median)" = moving_median(compliance),
  "compliance (moving mean)" = moving_mean(compliance),
  "compliance (local regression)" = local_regression(compliance, mt)
)

## Function first_of() helps to set NAs for specific phases.
## E.g., you want to replace the first two values of phase A and the first
## value of phase B and its preceding value.

byHeart2011 \%>\%
  transform(
    values = replace(values, first_of(phase == "A", 0:1), NA),
    values = replace(values, first_of(phase == "B", -1:0), NA)
  )
}
