% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/header_replication.R
\name{ar_ms}
\alias{ar_ms}
\title{Bayesian analysis of a Markov Switching autoregressive model}
\usage{
ar_ms(y, nlag = 1, beta_switch = FALSE, variance_switch = TRUE,
  identification_constraint = "variance", n_burn = 5000,
  n_rep = 20000, forecast_periods = 5, printout = FALSE,
  Hm1_delta = 25, mu_delta = 0, s_ = 0.3, nu_ = 3,
  R = matrix(c(8, 2, 2, 8), nrow = 2))
}
\arguments{
\item{y}{numeric vector (time series to be analyzed).}

\item{nlag}{integer, number of autoregressive lags (defaults to one).}

\item{beta_switch, variance_switch}{logicals, indicating whether there should be Markovian state 
dependence in regression parameters and residual variance, respectively. Defaults to 
\code{beta_switch = FALSE}, \code{variance_switch = TRUE}.}

\item{identification_constraint}{character, indicating how to identify latent states. Possible values:
\code{"variance"}, \code{"mean"} and \code{"persistence"}. Defaults to \code{"variance"}.}

\item{n_burn, n_rep}{integers, number of MCMC iterations for burn-in and main analysis.}

\item{forecast_periods}{number of future periods for which forecasts are computed.}

\item{printout}{logical, whether to print progress report during MCMC (defaults to \code{FALSE}).}

\item{Hm1_delta, mu_delta, s_, nu_, R}{prior parameters as described in KLTG (2019, Appendix E and Table 4).}
}
\value{
List containing parameter estimates and forecasts, with the following elements:
\itemize{
\item \code{pars}, matrix of posterior draws for parameters (rows are MCMC iterations, columns are parameters)
\item \code{fcMeans} and \code{fcSds}, matrices of forecast means and standard deviations (rows are MCMC iterations, columns are forecast horizons)
\item \code{probs}, matrix of filtered probabilities for first latent state (rows are MCMC iterations, columns are time periods, excluding the first \code{nlag} values for initialization). 
\item \code{count}, integer, counter for the number of states that were relabeled based on \code{identification_constraint}.
}
}
\description{
Bayesian analysis of a Markov Switching autoregressive model
}
\details{
The default parameters are as set by KLTG (2019, Section 5). The output matrices \code{fcMeans} and \code{fcSds} can be used to construct 
the mixture-of-parameters estimator analyzed by KLTG. While many of the model features can be changed as described above, the number of Markov regimes is always fixed at two. 

\link{ar_ms} is an R/C++ implementation of Matlab code kindly shared by Gianni Amisano via his website (\url{https://sites.google.com/site/gianniamisanowebsite/}). See Amisano and Giacomini (2007) who analyze a similar model.
}
\examples{
\dontrun{
# Use GDP data from 2014Q4 edition
data(gdp)
dat <- subset(gdp, vint == "2014Q4")
y <- dat$val[order(dat$dt)]

# Fit model, using the default settings 
set.seed(816)
fit <- ar_ms(y)

# Histograms of parameter draws
par(mfrow = c(2, 2))
hist(fit$pars[,1], main = "Intercept (state-invariant)", xlab = "")
hist(fit$pars[,2], main = "AR(1) term (state-invariant)", xlab = "")
hist(1/fit$pars[,3], main = "Residual variance in 1st state", xlab = "")
hist(1/fit$pars[,4], main = "Residual variance in 2nd state", xlab = "")

# By construction, the residual variance is smaller in the 1st than in the 2nd state:
print(mean(1/fit$pars[,3] < 1/fit$pars[,4]))
}
}
\references{
Amisano, G. and R. Giacomini (2007), `Comparing density forecasts via weighted likelihood ratio tests', Journal of Business and Economic Statistics 25, 177-190.

Krueger, F., Lerch, S., Thorarinsdottir, T.L. and T. Gneiting (2019): `Predictive inference based on Markov chain Monte Carlo output', working paper, Heidelberg Institute for Theoretical Studies, available at \url{http://arxiv.org/abs/1608.06802}.
}
\seealso{
\link{run_casestudy} uses \link{ar_ms} to replicate the results of KLTG (2019, Section 5).
}
\author{
Fabian Krueger, based on Matlab code by Gianni Amisano (see details section)
}
