% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tmb-sim.R
\name{simulate.sdmTMB}
\alias{simulate.sdmTMB}
\title{Simulate from a fitted sdmTMB model}
\usage{
\method{simulate}{sdmTMB}(
  object,
  nsim = 1L,
  seed = sample.int(1e+06, 1L),
  type = c("mle-eb", "mle-mvn"),
  model = c(NA, 1, 2),
  re_form = NULL,
  mcmc_samples = NULL,
  ...
)
}
\arguments{
\item{object}{sdmTMB model}

\item{nsim}{Number of response lists to simulate. Defaults to 1.}

\item{seed}{Random number seed}

\item{type}{How parameters should be treated. \code{"mle-eb"}: fixed effects
are at their maximum likelihood (MLE) estimates  and random effects are at
their empirical Bayes (EB) estimates. \code{"mle-mvn"}: fixed effects are at
their MLEs but random effects are taken from a single approximate sample.
This latter option is a suggested approach if these simulations will be
used for goodness of fit testing (e.g., with the DHARMa package).}

\item{model}{If a delta/hurdle model, which model to simulate from?
\code{NA} = combined, \code{1} = first model, \code{2} = second mdoel.}

\item{re_form}{\code{NULL} to specify a simulation conditional on fitted random
effects (this only simulates observation error). \code{~0} or \code{NA} to simulate
new random affects (smoothers, which internally are random effects, will
not be simulated as new).}

\item{mcmc_samples}{An optional matrix of MCMC samples. See \code{extract_mcmc()}
in the \href{https://github.com/pbs-assess/sdmTMBextra}{sdmTMBextra}
package.}

\item{...}{Extra arguments (not used)}
}
\value{
Returns a matrix; number of columns is \code{nsim}.
}
\description{
\code{simulate.sdmTMB} is an S3 method for producing a matrix of simulations from
a fitted model. This is similar to \code{\link[lme4:simulate.merMod]{lme4::simulate.merMod()}} and
\code{\link[glmmTMB:simulate.glmmTMB]{glmmTMB::simulate.glmmTMB()}}. It can be used with the \pkg{DHARMa} package
among other uses.
}
\examples{

# start with some data simulated from scratch:
set.seed(1)
predictor_dat <- data.frame(X = runif(300), Y = runif(300), a1 = rnorm(300))
mesh <- make_mesh(predictor_dat, xy_cols = c("X", "Y"), cutoff = 0.1)
dat <- sdmTMB_simulate(
  formula = ~ 1 + a1,
  data = predictor_dat,
  mesh = mesh,
  family = poisson(),
  range = 0.5,
  sigma_O = 0.2,
  seed = 42,
  B = c(0.2, -0.4) # B0 = intercept, B1 = a1 slope
)
fit <- sdmTMB(observed ~ 1 + a1, data = dat, family = poisson(), mesh = mesh)

# simulate from the model:
s1 <- simulate(fit, nsim = 300)
dim(s1)

# test whether fitted models are consistent with the observed number of zeros:
sum(s1 == 0)/length(s1)
sum(dat$observed == 0) / length(dat$observed)

# simulate with random effects sampled from their approximate posterior
s2 <- simulate(fit, nsim = 1, params = "mle-mvn")
# these may be useful in conjunction with DHARMa simulation-based residuals

# simulate with new random fields:
s3 <- simulate(fit, nsim = 1, re_form = ~ 0)
}
\seealso{
\code{\link[=sdmTMB_simulate]{sdmTMB_simulate()}}
}
