############################################################################################
## package 'secr'
## DENSITY.R
## 2010 04 29, 2010 05 01
## Write capture histories and traps to text files in DENSITY format
############################################################################################

write.capthist <- function (object, filestem = deparse(substitute(object)),
   sess = '1', ndec = 2, ...)

{
    sep <- list(...)$sep
    suffix <- '.txt'
    if(!is.null(sep))
        if (sep == ',')
            suffix <- '.csv'

    if (!is(object, 'capthist')) stop ('write.capthist requires a capthist object')
    det <- detector(traps(object))
    if (det %in% c('polygon','transect','signal','count','quadratbinary','quadratcount'))
        warning (paste('DENSITY cannot handle detector type',det))
    objectname <- deparse(substitute(object), control=NULL)

    if (filestem=='') {
        captfile <- ''
        trapfile <- ''
    }
    else {
        captfile <- paste(filestem, 'capt', suffix, sep='')
        trapfile <- paste(filestem, 'trap', suffix, sep='')
    }

    if (inherits(object, 'list')) {

        same <- all(sapply(traps(object)[-1], identical, traps(object)[[1]]))

        if (filestem != '') {
            if (same & !(filestem==''))
                trapfile <- paste(filestem, 'trap.txt', sep='')
            else
                trapfile <- paste(filestem, 'trap', session(object)[1], '.txt', sep='')
        }
        tempname <- ifelse(same, paste('traps(',objectname,')',sep=''),
            paste('traps(',objectname,') Session ',session(object)[1], sep=''))
        write.traps (traps(object)[[1]], file = trapfile, ..., header = tempname)
        write.captures (object[[1]], file = captfile, ..., deblank = TRUE,
            header = deparse(substitute(object), control=NULL), append = FALSE,
            sess = session(object[[1]]), ndec = ndec)
        for (i in 2:length(object)) {
            if (!same) {
                trapfile <- paste(filestem, 'trap', session(object)[i], suffix, sep='')
                write.traps (traps(object)[[i]], file = trapfile, ...)
            }
            write.captures (object[[i]], file = captfile, ..., deblank = TRUE,
                header = FALSE, append = TRUE, sess = session(object[[i]]), ndec = ndec)
        }
    }
    else {
        tempname <- paste('traps(',objectname,')',sep='')
        write.captures (object, file = captfile, ..., deblank = TRUE,
            header = deparse(substitute(object), control=NULL), append = FALSE,
            sess = session(object), ndec = ndec)
        write.traps (traps(object), file = trapfile, ..., header = tempname)
    }
}
############################################################################################

read.capthist <- function (captfile, trapfile, detector = 'multi', fmt = 'trapID',
                          noccasions = NULL, covnames = NULL, verify = TRUE, ...) {

    if (!(fmt %in% c('XY','trapID')))
        stop ('allowed formats are XY and trapID')
    if (length(captfile) != 1)
        stop ('require single captfile')
    replacedefaults <- function (default, user) replace(default, names(user), user)
    filetype <- function(x) {
        nx <- nchar(x)
        tolower(substring(x, nx-3, nx))
    }
    if ((filetype(captfile)=='.csv') & is.null(list(...)$sep))
        nfield <- max(count.fields(captfile, sep=',', ...))
    else
        nfield <- max(count.fields(captfile, ...))

    if (fmt == 'trapID') {
        nvar <- 4
        colcl <- c('character','character',NA,'character', rep(NA,nfield-nvar))
    }
    else {
        nvar <- 5
        colcl <- c('character','character',NA,NA,NA, rep(NA,nfield-nvar))
    }

    defaultdots <- list(sep = '', comment.char = '#')
    if (filetype(captfile)=='.csv') defaultdots$sep <- ','
    dots <- replacedefaults (defaultdots, list(...))
    capt <- do.call ('read.table', c(list(file = captfile, as.is = TRUE,
        colClasses = colcl), dots) )

    ## let's be clear about this...
    if (fmt =='trapID')
        names(capt)[1:4] <- c('Session','AnimalID','Occ','Trap')
    else
        names(capt)[1:5] <- c('Session','AnimalID','Occ','X','Y')
    if (any(is.na(capt[,1:nvar])))
        stop ('missing values not allowed')

    ## assumes file= is first argument of read.traps
    ## allows for multiple trap files
    defaultdots <- list(sep = '', comment.char = '#')
    if (filetype(trapfile[1])=='.csv') defaultdots$sep <- ','
    dots <- replacedefaults (defaultdots, list(...))
    readtraps <- function (x)
        do.call ('read.traps', c(list(file = x), detector = detector, dots) )
    trps <- sapply(trapfile, readtraps, simplify = FALSE)

    if (length(trps)==1) trps <- trps[[1]]
    temp <- make.capthist(capt, trps, fmt = fmt,  noccasions = noccasions,
        covnames = covnames, sortrows = TRUE)
    if (verify) verify(temp)
    temp
}

############################################################################################

