% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_measures_change_approx.R
\name{fit_measures_change_approx}
\alias{fit_measures_change_approx}
\title{Case Influence on Fit Measures (Approximate)}
\usage{
fit_measures_change_approx(
  fit,
  fit_measures = c("chisq", "cfi", "rmsea", "tli"),
  baseline_model = NULL,
  case_id = NULL,
  allow_inadmissible = FALSE,
  skip_all_checks = FALSE
)
}
\arguments{
\item{fit}{The output from \code{\link[lavaan:lavaan]{lavaan::lavaan()}} or its wrappers (e.g.,
\code{\link[lavaan:cfa]{lavaan::cfa()}} and \code{\link[lavaan:sem]{lavaan::sem()}}).}

\item{fit_measures}{The argument \code{fit.measures} used in
\link[lavaan:fitMeasures]{lavaan::fitMeasures}. Default is
\code{c("chisq", "cfi", "rmsea", "tli")}.}

\item{baseline_model}{The argument \code{baseline.model} used in
\link[lavaan:fitMeasures]{lavaan::fitMeasures}. Default is \code{NULL}.}

\item{case_id}{If it is a character vector of length equals to the
number of cases (the number of rows in the data in \code{fit}), then it
is the vector of case identification values. If it is \code{NULL}, the
default, then \code{case.idx} used by \code{lavaan} functions will be used
as case identification values.}

\item{allow_inadmissible}{If \code{TRUE}, accepts a fit object with
inadmissible results (i.e., \code{post.check} from
\code{\link[lavaan:lavInspect]{lavaan::lavInspect()}} is \code{FALSE}). Default is \code{FALSE}.}

\item{skip_all_checks}{If \code{TRUE}, skips all checks and allows
users to run this function on any object of \code{lavaan} class.
For users to experiment this and other functions on models
not officially supported. Default is \code{FALSE}.}
}
\value{
A matrix with the number of columns equals to the number of
requested fit measures, and the number of rows equals to the number
of cases. The row names are case identification values.
}
\description{
Gets a \code{\link[lavaan:lavaan]{lavaan::lavaan()}} output and computes the
approximate change
in selected fit measures if a case is included.
}
\details{
For each case, \code{\link[=fit_measures_change_approx]{fit_measures_change_approx()}} computes the
approximate differences in selected fit measures with and
without this case:

(Fit measure with all case) - (Fit measure without this case).

If the value of a case is positive, including the case increases an estimate.

If the value of a case is negative, including the case decreases an estimate.

Note that an increase is an improvement in fit for
goodness of fit measures such as CFI and TLI, but a decrease
is an improvement in fit for badness of fit measures such as
RMSEA and model chi-square.
This is a measure of the influence of a case on a fit measure
if it is included.

The model is not refitted. Therefore, the result is only an
approximation of that of \code{\link[=fit_measures_change]{fit_measures_change()}}. However, this
approximation is useful for identifying potentially influential
cases when the sample size is very large or the model takes a long
time to fit. This function can be used to identify potentially
influential cases quickly and then select them to conduct the
leave-one-out sensitivity analysis using \code{\link[=lavaan_rerun]{lavaan_rerun()}} and
\code{\link[=fit_measures_change]{fit_measures_change()}}.

For the technical details, please refer to the vignette
on this approach: \code{vignette("casewise_scores", package = "semfindr")}

Currently it only supports single-group models.
}
\examples{
library(lavaan)

# A path model

dat <- pa_dat
mod <-
"
m1 ~ a1 * iv1 + a2 * iv2
dv ~ b * m1
a1b := a1 * b
a2b := a2 * b
"
# Fit the model
fit <- lavaan::sem(mod, dat)
summary(fit)

# Approximate changes
out_approx <- fit_measures_change_approx(fit, fit_measures = "chisq")
head(out_approx)
# Fit the model several times. Each time with one case removed.
# For illustration, do this only for four selected cases
fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = 1:5)
# Compute the changes in chisq if a case is included
# vs. if this case is excluded.
# That is, case influence on model chi-squared.
out <- fit_measures_change(fit_rerun, fit_measures = "chisq")
# Case influence, for the first few cases
head(out)
# Compare the results
plot(out_approx[1:5, "chisq"], out)
abline(a = 0, b = 1)

# A CFA model

dat <- cfa_dat
mod <-
"
f1 =~  x1 + x2 + x3
f2 =~  x4 + x5 + x6
f1 ~~ f2
"
# Fit the model
fit <- lavaan::cfa(mod, dat)

out_approx <- fit_measures_change_approx(fit, fit_measures = "chisq")
head(out_approx)

fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = 1:5)
# Compute the changes in chisq if a case is included
# vs. if this case is excluded.
# That is, case influence on fit measures.
out <- fit_measures_change(fit_rerun, fit_measures = "chisq")
# Results excluding a case, for the first few cases
head(out)
# Compare the results
plot(out_approx[1:5, "chisq"], out)
abline(a = 0, b = 1)

# A latent variable model

dat <- sem_dat
mod <-
"
f1 =~  x1 + x2 + x3
f2 =~  x4 + x5 + x6
f3 =~  x7 + x8 + x9
f2 ~   a * f1
f3 ~   b * f2
ab := a * b
"
# Fit the model
fit <- lavaan::sem(mod, dat)

out_approx <- fit_measures_change_approx(fit, fit_measures = "chisq")
head(out_approx)

fit_rerun <- lavaan_rerun(fit, parallel = FALSE,
                          to_rerun = 1:5)
# Compute the changes in chisq if a case is excluded
# vs. if this case is included.
# That is, case influence on model chi-squared.
out <- fit_measures_change(fit_rerun, fit_measures = "chisq")
# Case influence, for the first few cases
head(out)
# Compare the results
plot(out_approx[1:5, "chisq"], out)
abline(a = 0, b = 1)

}
\author{
Idea by Mark Hok Chio Lai \url{https://orcid.org/0000-0002-9196-7406},
implemented by Shu Fai Cheung \url{https://orcid.org/0000-0002-9871-9448}.
}
