% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/s2_rgb.R
\name{s2_rgb}
\alias{s2_rgb}
\title{Create RGB images from S2 reflectance products.}
\usage{
s2_rgb(
  infiles,
  prod_type = NA,
  rgb_bands = 4:2,
  scaleRange = NA,
  outdir = NA,
  subdirs = NA,
  format = NA,
  compress = "DEFLATE",
  bigtiff = FALSE,
  tmpdir = NA,
  rmtmp = TRUE,
  proc_mode = "raster",
  parallel = TRUE,
  overwrite = FALSE,
  .log_message = NA,
  .log_output = NA
)
}
\arguments{
\item{infiles}{A vector of input filenames. Input files are paths
of products already converted from SAFE format to a
format managed by GDAL (use \link{s2_translate} to do it);
their names must be in the sen2r naming convention
(\link{safe_shortname}).}

\item{prod_type}{(optional) Output product (see \link{safe_shortname} for the
list of accepted products). If not provided, it is retrieved from the
file name.}

\item{rgb_bands}{(optional) 3-length integer vector, which the number of the
bands to be used respectively for red, green and blue.
Default is 4:2 (true colours).
It is also possible to pass a list of 3-length integer vectors
in order to create multiple RGB types for each input file.
Notice that this is the \href{https://sentinels.copernicus.eu/web/sentinel/user-guides/sentinel-2-msi/resolutions/spatial}{actual number name of the bands}:
so, to use i.e. BOA band 11 (1610nm) use the number 11, even if band 11 is
the 10th band of a BOA product (because band 10 is missing).}

\item{scaleRange}{(optional) Range of valid values. If can be a 2-length
integer vector (min-max for all the 3 bands) or a 6-length vector or
3x2 matrix (min red, min green, min blue, max red, max green, max blue).
Default is to use c(0,2500) for bands 1-5; c(0,7500) bands 6-12.}

\item{outdir}{(optional) Full name of the existing output directory
where the files should be created. Default is the same directory of
input reflectance files.}

\item{subdirs}{(optional) Logical: if TRUE, different indices are
placed in separated \code{outfile} subdirectories; if FALSE, they are placed in
\code{outfile} directory; if NA (default), subdirectories are created only if
more than a single spectral index is required.}

\item{format}{(optional) Format of the output file (in a
format recognised by GDAL). Default is the same format of input images
(or "GTiff" in case of VRT input images).}

\item{compress}{(optional) In the case a GTiff format is
present, the compression indicated with this parameter is used.}

\item{bigtiff}{(optional) Logical: if TRUE, the creation of a BigTIFF is
forced (default is FALSE).
This option is used only in the case a GTiff format was chosen.}

\item{tmpdir}{(optional) Path where intermediate files (VRT) will be created.
Default is a temporary directory.
If \code{tmpdir} is a non-empty folder, a random subdirectory will be used.}

\item{rmtmp}{(optional) Logical: should temporary files be removed?
(Default: TRUE)}

\item{proc_mode}{(optional) Character: if \code{"gdal_calc"},
\code{gdal_calc} routines are used to compute indices;
if \code{"raster"} or \code{"stars"}, R functions are instead used
(using respectively \code{raster} or \code{stars} routines).
\strong{Note}: default value (\code{"raster"}) is the only fully supported mode.
\code{"gdal_calc"} can be used only if a runtime GDAL environment can be properly
configured (no assistance is provided in case of GDAL-related problems).
\code{"raster"} mode is experimental.
See \code{s2_calcindices()} for further details.}

\item{parallel}{(optional) Logical: if TRUE, the function is run using parallel
processing, to speed-up the computation for large rasters.
The number of cores is automatically determined; specifying it is also
possible (e.g. \code{parallel = 4}).
If FALSE (default), single core processing is used.}

\item{overwrite}{(optional) Logical value: should existing thumbnails be
overwritten? (default: TRUE)}

\item{.log_message}{(optional) Internal parameter
(it is used when the function is called by \code{sen2r()}).}

\item{.log_output}{(optional) Internal parameter
(it is used when the function is called by \code{sen2r()}).}
}
\value{
A vector with the names of the created images.
}
\description{
Function to create RGB images from Sentinel-2 reflectances.
}
\note{
License: GPL 3.0
}
\examples{
\donttest{
# Define file names
ex_in <- system.file(
  "extdata/out/S2A2A_20190723_022_Barbellino_BOA_10.tif",
  package = "sen2r"
)

# Run function
ex_out <- s2_rgb(
  infiles = ex_in,
  rgb_bands = list(c(11,8,4),c(9,5,4)),
  scaleRange = list(c(0,7500), matrix(c(rep(0,3),8500,6000,4000),ncol=2)),
  outdir = tempdir(),
  compress = 50
)
ex_out

# Show output
oldpar <- par(mfrow = c(1,3), mar = rep(0,4))
image(stars::read_stars(ex_in), rgb = 4:2, maxColorValue = 3500, useRaster = TRUE)
image(stars::read_stars(ex_out[1]), rgb = 1:3, useRaster = TRUE)
image(stars::read_stars(ex_out[2]), rgb = 1:3, useRaster = TRUE)
par(oldpar)
}
}
\references{
L. Ranghetti, M. Boschetti, F. Nutini, L. Busetto (2020).
"sen2r": An R toolbox for automatically downloading and preprocessing
Sentinel-2 satellite data. \emph{Computers & Geosciences}, 139, 104473.
\doi{10.1016/j.cageo.2020.104473}, URL: \url{https://sen2r.ranghetti.info/}.
}
\author{
Luigi Ranghetti, phD (2019) \email{luigi@ranghetti.info}
}
