% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sgo_bng.R
\encoding{UTF-8}
\name{sgo_lonlat_bng}
\alias{sgo_lonlat_bng}
\title{GCS to BNG Easting/Northing}
\usage{
sgo_lonlat_bng(x, to=27700, OSTN=TRUE, OD=FALSE)
}
\arguments{
\item{x}{A \code{sgo_points} object with coordinates defined in a Geodetic
Coordinate System expressed as Longitude and Latitude (e.g. epsg=4258, 4937,
4326, 4979 or 4277)}

\item{to}{Specifies the EPSG code to convert the coordinates to. It can only
take the following values: \code{27700} or \code{7405}.}

\item{OSTN}{Logical variable indicating whether use OSTN15 transformation
when TRUE or a less accurate but slightly faster single Helmert
transformation when FALSE.}

\item{OD}{Logical variable. When TRUE, and the output contains a
column with heights, then a new column is added to the result indicating the
ordnance datum (OD) used on each point. It is ignored when \code{OSTN=FALSE}.}
}
\value{
An object of class \code{sgo_points} whose coordinates are defined as
Easting/Northing (epsg=27700 or 7405). They are adjusted to the SW corner of
1m grid square. If \code{OD=TRUE} a column named \code{height.datum} is
added to the resulting object.
}
\description{
Converts a geodetic coordinate system to BNG (projected) Easting/Northing
coordinates. It also transforms GPS ellipsoid heights to orthometric
(mean sea level) heights on the relevant Ordnance Survey mapping datum, using
the National Geoid Model OSGM15.
}
\details{
The UK Ordnance Survey defined 'OSGB36' as the datum for the UK, based on the
'Airy 1830' ellipsoid. However, in 2014, they deprecated OSGB36 in favour of
ETRS89 for longitude/latitude coordinates. Thus, \code{x} should be defined
as ETRS89 (or WGS84) most of the times.

Note: When transforming from EPSG=4277 any height included in the input
will be simply discarded (see \code{\link{sgo_points}}).

According to the Transformations and OSGM15 User Guide, p. 8:
\emph{"...ETRS89 is a precise version of the better known WGS84 reference
system optimised for use in Europe; however, for most purposes it can be
considered equivalent to WGS84."} and \emph{"For all navigation, mapping,
GIS, and engineering applications within the tectonically stable parts of
Europe (including UK and Ireland), the term ETRS89 should be taken as
synonymous with WGS84."} This means that EPSGs with the ETRS89 datum or
WGS84 will be considered equivalent by this routine.

\strong{Note}: All those coordinates outside the rectangle covered by OSTN15
will be automatically computed using the small Helmert transformation. Such
coordinates will be accurate up to about +/-5 metres, therefore the resulting
easting and northing coordinates will be rounded to the metre. Since those
coordinates are outside of the OSTN15 domain the resulting coordinates will
have the resulting height defined as \code{NA}.
Similarly, when using \code{OSTN=FALSE} on 3D coordinates, the result will
have all the heights defined as \code{NA}.
Converting from lon/lat to BNG coordinates is faster than the other way
around, due to the iterative nature of the algorithm that is built into
OSTN15.
}
\examples{
lon <- c(-4.25181,-3.18827)
lat <- c(55.86424, 55.95325)
pts <- sgo_points(list(longitude=lon, latitude=lat), epsg=4326)
bng.pts <- sgo_lonlat_bng(pts)
}
\references{
Ordnance Survey Limited, 2018. \emph{Transformations and OSGM15 user guide}
}
\seealso{
\code{\link{sgo_points}}, \code{\link{sgo_bng_lonlat}},
\code{\link{sgo_set_gcs}}, \code{\link{sgo_transform}}.
}
