% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sgpvalue.R
\name{sgpvalue}
\alias{sgpvalue}
\title{Second-Generation p-Values}
\usage{
sgpvalue(
  est.lo,
  est.hi,
  null.lo,
  null.hi,
  inf.correction = 1e-05,
  warnings = TRUE
)
}
\arguments{
\item{est.lo}{A numeric vector of lower bounds of interval estimates. Values may be finite or \code{-Inf} or \code{Inf}. Must be of same length as \code{est.hi}.}

\item{est.hi}{A numeric vector of upper bounds of interval estimates. Values may be finite or \code{-Inf} or \code{Inf}. Must be of same length as \code{est.lo}.}

\item{null.lo}{A numeric vector of lower bounds of null intervals. Values may be finite or \code{-Inf} or \code{Inf}. Must be of same length as \code{null.hi}.}

\item{null.hi}{A numeric vector of upper bounds of null intervals. Values may be finite or \code{-Inf} or \code{Inf}. Must be of same length as \code{null.lo}.}

\item{inf.correction}{A small scalar to denote a positive but infinitesimally small SGPV. Default is 1e-5. SGPVs that are infinitesimally close to 1 are assigned \code{1-inf.correction}. This option can only be invoked when one of the intervals has infinite length.}

\item{warnings}{Warnings toggle. Warnings are on by default.}
}
\value{
A list containing the following components:
\describe{
\item{\code{p.delta}}{Vector of second-generation p-values}
\item{\code{delta.gap}}{Vector of delta-gaps. Reported as \code{NA} when the corresponding second-generation p-value is not zero.}
}
}
\description{
This function computes the second-generation \emph{p}-value (SGPV) and its associated delta gaps, as introduced in Blume et al. (2018).
}
\details{
Values of \code{NA} or \code{NaN} for \code{est.lo}, \code{est.hi}, \code{null.lo}, or \code{null.lo} will yield a warning and result in a SGPV of \code{NA} or \code{NaN}.

When \code{null.hi} and \code{null.lo} are of length 1, the same null interval is used for every interval estimate of [\code{est.lo}, \code{est.hi}]. If \code{null.hi} is not of length 1, its length must match that of \code{est.hi}.

When possible, one should compute the second-generation \emph{p}-value on a scale that is symmetric about the null hypothesis. For example, if the parameter of interest is an odds ratio, computations are typically done on the log scale. This keeps the magnitude of positive and negative delta-gaps comparable. Also, recall that the delta-gaps magnitude is not comparable across different null intervals.
}
\examples{

## Simple example for three estimated log odds ratios but the same null interval
lb <- c(log(1.05), log(1.3), log(0.97))
ub <- c(log(1.8), log(1.8), log(1.02))
sgpv <- sgpvalue(est.lo = lb, est.hi = ub, null.lo = log(1/1.1), null.hi = log(1.1))
sgpv$p.delta

sgpv$delta.gap

## Works with infinte interval bounds
sgpvalue(est.lo = log(1.3), est.hi = Inf, null.lo = -Inf, null.hi = log(1.1))


sgpvalue(est.lo = log(1.05), est.hi = Inf, null.lo = -Inf, null.hi = log(1.1))



## Example t-test with simulated data
set.seed(1776)
x1 <- rnorm(15,mean=0,sd=2) ; x2 <- rnorm(15,mean=3,sd=2)
ci <- t.test(x1,x2)$conf.int[1:2]
sgpvalue(est.lo = ci[1], est.hi = ci[2], null.lo = -1, null.hi = 1)

set.seed(2019)
x1 <- rnorm(15,mean=0,sd=2) ; x2 <- rnorm(15,mean=3,sd=2)
ci <- t.test(x1,x2)$conf.int[1:2]
sgpvalue(est.lo = ci[1], est.hi = ci[2], null.lo = -1, null.hi = 1)

## Simulated two-group dichotomous data for different parameters
set.seed(1492)
n1 <- n2 <- 30
x1 <- rbinom(1,size=n1,p=0.15) ; x2 <- rbinom(1,size=n2,p=0.50)

# On the difference in proportions
ci.p  <- prop.test(c(x1,x2),n=c(n1,n2))$conf.int[1:2]
sgpvalue(est.lo = ci.p[1], est.hi = ci.p[2], null.lo = -0.2, null.hi = 0.2)

# On the log odds ratio scale
a <- x1 ; b <- x2 ; c <- n1-x1 ; d <- n2-x2
ci.or <- log(a*d/(b*c)) + c(-1,1)*1.96*sqrt(1/a+1/b+1/c+1/d)	# Delta-method SE for log odds ratio
sgpvalue(est.lo = ci.or[1], est.hi = ci.or[2], null.lo = log(1/1.5), null.hi = log(1.5))


}
\references{
Blume JD, Greevy RA Jr., Welty VF, Smith JR, Dupont WD (2019). An Introduction to Second-generation \emph{p}-values. \emph{The American Statistician}. 73:sup1, 157-167, DOI: https://doi.org/10.1080/00031305.2018.1537893

Blume JD, D’Agostino McGowan L, Dupont WD, Greevy RA Jr. (2018). Second-generation \emph{p}-values: Improved rigor, reproducibility, & transparency in statistical analyses. \emph{PLoS ONE} 13(3): e0188299. https://doi.org/10.1371/journal.pone.0188299
}
\seealso{
\code{\link{fdrisk}, \link{sgpower}, \link{plotsgpv}}
}
