#' Function to call a method to update \code{plotly} traces
#'
#' @description
#' It is used by registering in a \code{shiny} application.
#' It receives events in \code{plotly} figure and update it using a method of
#' a \code{downsampler} instance.
#' See the examples in \code{downsampler} class.
#' @param session \code{session} object.
#' The object passed to function given to \code{shinyServer}.
#' @param outputId Character.
#' The \code{outputId} of the figure, whose data will be down-sampled
#' @param relayout_order Named list.
#' The list generated by \code{plotlyjs_relayout},
#' which is obtained using \code{plotly::event_data}.
#' @param ds_obj \code{downsampler} instance.
#' The instance containing original data of the figure,
#' which is also used for updating the traces of \code{plotly}.
#' @param reset Boolean.
#' It it is \code{TRUE}, the figure will be updated even if
#' \code{relayout_order} is \code{NULL}.
#' The ranges of x axes are reset (initialized).
#' @param reload Boolean.
#' It it is \code{TRUE}, the figure will be updated even if
#' \code{relayout_order} is \code{NULL}.
#' The ranges of x axes are preserved.
#' @param verbose Boolean.
#' Whether detailed messages to check the procedures are shown. By default, \code{FALSE}.
#'
#' @export
#'
updatePlotlyH <- function(
    session, outputId, relayout_order, ds_obj,
    reset = FALSE, reload = FALSE, verbose = FALSE
    ) {

  if (verbose) {
    message("Check the data type of the downsampler object (updatePlotlyH)")
  }
  assertthat::assert_that(inherits(ds_obj, "downsampler"))

  if (verbose) {
    message("Get updated traces using downsampler$update_trace (updatePlotlyH)")
  }

  trace_update_order <- ds_obj$update_trace(
    relayout_order = relayout_order,
    reset = reset, reload = reload, send_trace = TRUE
    )

  if (!is.null(trace_update_order) && length(trace_update_order$new_trace) > 0) {

    if (verbose) {
      message("Updated the traces using plotlyProxy (updatePlotlyH)")
    }

    plotlyProxy(outputId, session) %>%
      plotlyProxyInvoke(
        "deleteTraces", unique(trace_update_order$trace_idx_update) - 1
      ) %>%
      plotlyProxyInvoke(
        "addTraces",
        trace_update_order$new_trace
      )
  } else {

    if (verbose) {
      message("The update was NULL (updatePlotlyH)")
    }
  }
}
