% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cd_sm_uncured.R
\name{cd.sm.uncured}
\alias{cd.sm.uncured}
\title{Smoothed cross-validation conditional density estimator of the susceptible population}
\usage{
cd.sm.uncured(x, time, delta, logh3, logh4, k = 10)
}
\arguments{
\item{x}{A numeric vector giving the covariate values.}

\item{time}{A numeric vector giving the observed times.}

\item{delta}{A numeric vector giving the values of the uncensoring indicator, where 1 indicates that the event of interest has been observed and 0 indicates that the observation is censored.}

\item{logh3}{The logarithm of the bandwidth for smoothing the time variable.}

\item{logh4}{The logarithm of the bandwidth for smoothing the covariate.}

\item{k}{The number of nearest neighbors used to smooth.}
}
\value{
A vector containing the cross-validation conditional density of the susceptible population for each observation \eqn{(X_i, T_i)}, smoothed by considering the \code{k} nearest neighbors with Mahalanobis distance.
}
\description{
This function implements a smoothed version of the nonparametric cross-validation estimator for the conditional density  of the susceptible population proposed by Piñeiro-Lamas (2024) (see Equation (3.5)).
Smoothing is done using the \code{k} nearest neighbors based on Mahalanobis distance.
The Mahalanobis distance between two points \eqn{(X_i, T_i)} and \eqn{(X_j, T_j)} is defined as:
\deqn{ d_M((X_i, T_i), (X_j, T_j)) = \sqrt{ \left( \begin{pmatrix} X_i \\ T_i \end{pmatrix} - \begin{pmatrix} X_j \\ T_j \end{pmatrix} \right)^t \Sigma^{-1} \left( \begin{pmatrix} X_i \\ T_i \end{pmatrix} - \begin{pmatrix} X_j \\ T_j \end{pmatrix} \right) }, }
where \eqn{\Sigma} is the covariance matrix of the joint distribution of \eqn{(X, T)}.
}
\examples{
# Some artificial data
set.seed(123)
n <- 50
x <- runif(n, -2, 2) # Covariate values
y <- rweibull(n, shape = 0.5 * (x + 4)) # True lifetimes
c <- rexp(n) # Censoring values
p <- exp(2*x)/(1 + exp(2*x)) # Probability of being susceptible
u <- runif(n)
t  <- ifelse(u < p, pmin(y, c), c) # Observed times
d  <- ifelse(u < p, ifelse(y < c, 1, 0), 0) # Uncensoring indicator
data <- data.frame(x = x, t = t, d = d)
suppressWarnings(cd.sm.uncured(x, t, d, log(0.5), log(0.5), k=10))
}
\references{
Mahalanobis, P. C. (1936). On the generalised distance in statistics. Proceedings of the National Institute of Sciences of India, 2, 49-55.

Piñeiro-Lamas, B. (2024). High dimensional single-index mixture cure models [PhD thesis]. Universidade da Coruña. Available at \url{https://ruc.udc.es/dspace/handle/2183/37035}
}
\seealso{
\code{\link[sicure]{cd.uncured}}, \code{\link[sicure]{kNN.Mahalanobis}}
}
