#' Show Simplified Signature Number Survey
#'
#' [sig_estimate] shows comprehensive rank survey generated by
#' **NMF** package, sometimes
#' it is hard to consider all measures. Here provides a
#' one or two y-axis visualization method to help users determine
#' the optimal signature number (showing both
#' stability ("cophenetic") and error (RSS) at default).
#' Users can also set custom measures to show.
#'
#' @param object a `Survey` object generated from [sig_estimate], or
#' a `data.frame` contains at least rank columns and columns for
#' one measure.
#' @param x column name for x axis.
#' @param left_y column name for left y axis.
#' @param right_y column name for right y axis.
#' @param left_name label name for left y axis.
#' @param right_name label name for right y axis.
#' @param left_color color for left axis.
#' @param right_color color for right axis.
#' @param left_shape,right_shape,shape_size shape setting.
#' @return a `ggplot` object
#' @export
#'
#' @examples
#' \donttest{
#' load(system.file("extdata", "toy_copynumber_tally_M.RData",
#'   package = "sigminer", mustWork = TRUE
#' ))
#' library(NMF)
#' cn_estimate <- sig_estimate(cn_tally_M$nmf_matrix,
#'   cores = 1, nrun = 5,
#'   verbose = TRUE
#' )
#'
#' # Show two measures
#' show_sig_number_survey(cn_estimate)
#' # Show one measure
#' p <- show_sig_number_survey(cn_estimate, right_y = NULL)
#' p
#' p <- add_h_arrow(p, x = 4.1, y = 0.953, label = "selected number")
#' p
#'
#' # Show data from a data.frame
#' show_sig_number_survey(cn_estimate$survey)
#' # Show other measures
#' head(cn_estimate$survey)
#' show_sig_number_survey(cn_estimate$survey,
#'   right_y = "dispersion",
#'   right_name = "dispersion"
#' )
#' show_sig_number_survey(cn_estimate$survey,
#'   right_y = "evar",
#'   right_name = "evar"
#' )
#' }
#' @tests
#' load(system.file("extdata", "toy_copynumber_tally_M.RData",
#'   package = "sigminer", mustWork = TRUE
#' ))
#' library(NMF)
#' cn_estimate <- sig_estimate(cn_tally_M$nmf_matrix,
#'   cores = 1, nrun = 5,
#'   verbose = TRUE
#' )
#' expect_equal(length(cn_estimate), 2)
#'
#' # Show two measures
#' p = show_sig_number_survey(cn_estimate)
#' expect_s3_class(p, "ggplot")
#'
#' # Show one measure
#' p = show_sig_number_survey(cn_estimate, right_y = NULL)
#' expect_s3_class(p, "ggplot")
#' p = add_h_arrow(p, x = 4.1, y = 0.953, label = "selected number")
#' expect_s3_class(p, "ggplot")
#'
#' # Show data from a data.frame
#' p = show_sig_number_survey(cn_estimate$survey)
#' expect_s3_class(p, "ggplot")
#'
#' # Show other measures
#' head(cn_estimate$survey)
#' p = show_sig_number_survey(cn_estimate$survey,
#'   right_y = "dispersion",
#'   right_name = "dispersion"
#' )
#' expect_s3_class(p, "ggplot")
#' p = show_sig_number_survey(cn_estimate$survey,
#'   right_y = "evar",
#'   right_name = "evar"
#' )
#' expect_s3_class(p, "ggplot")
#'
#' @seealso [sig_estimate] for estimating signature number for [sig_extract],
#' [show_sig_number_survey2] for more visualization method.
show_sig_number_survey <- function(object, x = "rank",
                                   left_y = "cophenetic", right_y = "rss",
                                   left_name = left_y, right_name = toupper(right_y),
                                   left_color = "black", right_color = "red",
                                   left_shape = 16, right_shape = 18, shape_size = 4) {
  stopifnot(class(object) == "Survey" | is.data.frame(object))
  if (class(object) == "Survey") {
    survey <- object$survey
  } else {
    survey <- object
  }

  if (is.null(right_y)) {
    # Show one-axis plot
    p <- ggplot(data = survey) +
      geom_point(aes_string(x = x, y = left_y), color = left_color, shape = left_shape, size = shape_size) +
      geom_line(aes_string(x = x, y = left_y), color = left_color) +
      scale_x_continuous(breaks = unique(survey[[x]]), labels = unique(survey[[x]])) +
      scale_y_continuous(name = left_name) +
      cowplot::theme_cowplot() +
      xlab("Number of signature")
  } else {
    # Show two-axis plot
    survey$new_right <- norm2rg(
      survey[[right_y]],
      range(survey[[left_y]])
    )

    p <- ggplot(data = survey) +
      geom_point(aes_string(x = x, y = left_y), color = left_color, shape = left_shape, size = shape_size) +
      geom_point(aes_string(x = x, y = "new_right"), color = right_color, shape = right_shape, size = shape_size) +
      geom_line(aes_string(x = x, y = left_y), color = left_color) +
      geom_line(aes_string(x = x, y = "new_right"), color = right_color) +
      scale_x_continuous(breaks = unique(survey[[x]]), labels = unique(survey[[x]])) +
      scale_y_continuous(
        name = left_name,
        sec.axis = sec_axis(~ norm2rg(., range(survey[[right_y]])),
          name = right_name
        )
      ) +
      cowplot::theme_cowplot() +
      xlab("Number of signature") +
      theme(
        axis.title.y.left = element_text(color = left_color),
        axis.text.y.left = element_text(color = left_color),
        axis.title.y.right = element_text(color = right_color),
        axis.text.y.right = element_text(color = right_color),
        panel.border = element_rect(colour = "black", fill = NA, size = 1)
      )
  }

  return(p)
}


# Normalize data to specified range
norm2rg <- function(x, rg) {
  stopifnot(length(rg) == 2)
  k <- diff(rg) / diff(range(x))
  rg[1] + k * (x - min(x))
}
