% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frq.R
\name{frq}
\alias{frq}
\title{Frequency table of labelled variables}
\usage{
frq(x, ..., sort.frq = c("none", "asc", "desc"), weight.by = NULL,
  auto.grp = NULL, show.strings = TRUE, grp.strings = NULL,
  out = c("txt", "viewer", "browser"))
}
\arguments{
\item{x}{A vector or a data frame. May also be a grouped data frame
(see 'Note' and 'Examples').}

\item{...}{Optional, unquoted names of variables that should be selected for
further processing. Required, if \code{x} is a data frame (and no
vector) and only selected variables from \code{x} should be processed.
You may also use functions like \code{:} or tidyselect's
\code{\link[tidyselect]{select_helpers}}.
See 'Examples' or \href{../doc/design_philosophy.html}{package-vignette}.}

\item{sort.frq}{Determines whether categories should be sorted
according to their frequencies or not. Default is \code{"none"}, so
categories are not sorted by frequency. Use \code{"asc"} or
\code{"desc"} for sorting categories ascending or descending order.}

\item{weight.by}{Bare name, or name as string, of a variable in \code{x}
that indicates the vector of weights, which will be applied to weight all
observations. Default is \code{NULL}, so no weights are used.}

\item{auto.grp}{Numeric value, indicating the minimum amount of unique
values in a variable, at which automatic grouping into smaller  units
is done (see \code{\link{group_var}}). Default value for \code{auto.group}
is \code{NULL}, i.e. auto-grouping is off.}

\item{show.strings}{Logical, if \code{TRUE}, frequency tables for character
vectors will not be printed. This is useful when printing frequency tables
of all variables from a data frame, and due to computational reasons
character vectors should not be printed.}

\item{grp.strings}{Numeric, if not \code{NULL}, groups string values in
character vectors, based on their similarity. The similarity is estimated
with the \pkg{stringdist}-package. See \code{\link{group_str}} for details
on grouping, and that function's \code{maxdist}-argument to get more
details on the distance of strings to be treated as equal.}

\item{out}{Character vector, indicating whether the results should be printed
to console (\code{out = "txt"}) or as HTML-table in the viewer-pane
(\code{out = "viewer"}) or browser (\code{out = "browser"}).}
}
\value{
A list of data frames with values, value labels, frequencies, raw, valid and
          cumulative percentages of \code{x}.
}
\description{
This function returns a frequency table of labelled vectors, as data frame.
}
\note{
\code{x} may also be a grouped data frame (see \code{\link[dplyr]{group_by}})
      with up to two grouping variables. Frequency tables are created for each
      subgroup then.
      \cr \cr
      The \code{print()}-method adds a table header with information on the
      variable label, variable type, total and valid N, and mean and
      standard deviations. Mean and SD are \emph{always} printed, even for
      categorical variables (factors) or character vectors. In this case,
      values are coerced into numeric vector to calculate the summary
      statistics.
}
\examples{
library(haven)
# create labelled integer
x <- labelled(
  c(1, 2, 1, 3, 4, 1),
  c(Male = 1, Female = 2, Refused = 3, "N/A" = 4)
)
frq(x)

x <- labelled(
  c(1:3, tagged_na("a", "c", "z"), 4:1, 2:3),
  c("Agreement" = 1, "Disagreement" = 4, "First" = tagged_na("c"),
    "Refused" = tagged_na("a"), "Not home" = tagged_na("z"))
)
frq(x)

# in a pipe
data(efc)
library(dplyr)
efc \%>\%
  select(e42dep, e15relat, c172code) \%>\%
  frq()

# or:
# frq(efc, e42dep, e15relat, c172code)

# with grouped data frames, in a pipe
efc \%>\%
  group_by(e16sex, c172code) \%>\%
  frq(e16sex, c172code, e42dep)

# with select-helpers: all variables from the COPE-Index
# (which all have a "cop" in their name)
frq(efc, contains("cop"))

# all variables from column "c161sex" to column "c175empl"
frq(efc, c161sex:c175empl)

# for non-labelled data, variable name is printed,
# and "label" column is removed from output
data(iris)
frq(iris, Species)

# group variables with large range
frq(efc, c160age)
frq(efc, c160age, auto.grp = 5)

# and with weights
efc$weights <- abs(rnorm(n = nrow(efc), mean = 1, sd = .5))
frq(efc, c160age, auto.grp = 5, weight.by = weights)

# group string values
\dontrun{
dummy <- efc \%>\% dplyr::select(3)
dummy$words <- sample(
  c("Hello", "Helo", "Hole", "Apple", "Ape",
    "New", "Old", "System", "Systemic"),
  size = nrow(dummy),
  replace = TRUE
)

frq(dummy)
frq(dummy, grp.strings = 2)}

}
\seealso{
\code{\link{flat_table}} for labelled (proportional) tables.
}
