#' Generate Main Data
#'
#' Given the league ID, generate a main data set that can be used for
#' generating quick insights. This includes teams, fantasy points,
#' avatars, divisions, owner IDs, and more.
#'
#' @return Returns a data frame containing roster, league user,
#'         and overall league information.
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, December 2023
#' @keywords roster league user full main
#' @importFrom dplyr left_join rename bind_rows
#' @export
#' @examples
#' \dontrun{get_main_data(688281863499907072)}
#'
#' @param league_id League ID generated by Sleeper (numeric).
#'
get_main_data <- function(league_id) {
  # Query league data to see if league ID is valid
  league_data <- suppressMessages(get_league(league_id))
  # Check if league data is null
  if (is.null(league_data)) {
    # If NULL, inform user and return nothing
    message("League ID did not return any results. Did you enter the league ID correctly?") # nolint
  } else {
    # If not null, query data with league ID to get roster and league user data
    roster_data <- get_rosters(league_id)
    league_user_data <- get_league_users(league_id)
    # Remove league ID from league user data (already present in roster data)
    league_user_data$league_id <- NULL
    # Rename avatar column (naming convention present in other data)
    league_data <- dplyr::rename(league_data, "avatar_league" = "avatar")
    # Join roster data to league user data
    master_df_pre <- dplyr::left_join(roster_data,
                                      league_user_data,
                                      by = c("owner_id" = "user_id"))
    # Join previous data frame to league data data frame
    master_df <- dplyr::left_join(master_df_pre,
                                  league_data,
                                  by = "league_id")
    # Add total fantasy points (for and against) columns
    master_df$fpts_total <- master_df$fpts + master_df$fpts_decimal / 100
    master_df$fpts_against_total <- master_df$fpts_against +
      master_df$fpts_against_decimal / 100
    # Add streak ranking column
    master_df$streak_ranking <- ifelse(substr(master_df$streak,
                                              start = 2,
                                              stop = 2) == "L",
                                       -as.numeric(substr(master_df$streak,
                                                          start = 1,
                                                          stop = 1)),
                                       as.numeric(substr(master_df$streak,
                                                         start = 1,
                                                         stop = 1)))
    # Get the number of divisions
    num_divisions <- max(master_df$division)
    # Construct table for matching division number and division name
    division_df <- data.frame()
    for (i in 1:num_divisions) {
      division_name <- unique(master_df[, paste0("division_", i)])
      temp_df <- data.frame(division = c(i),
                            division_name = c(division_name))
      division_df <- dplyr::bind_rows(division_df, temp_df)
    }
    # Add division names column (currently only goes up to four)
    master_df <- dplyr::left_join(master_df, division_df, by = "division")
    # Add ranking based on wins/losses and fantasy points for
    master_df$rank_fpts <- rank(-master_df$wins * 400
                                + master_df$losses * 400
                                - master_df$fpts_total)
    # Return full data frame
    return(master_df)
  }
}